/* Various utilities
   Copyright (C) 1994 Miguel de Icaza, Janne Kukonlehto, Mauricio Plaza.

   The file_date routine is mostly from GNU's fileutils package,
   written by Richard Stallman and David MacKenzie.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <signal.h>		/* my_system */
#include <limits.h>		/* INT_MAX */
#include <sys/time.h>		/* select: timeout */
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdarg.h>
#include <sys/wait.h>		/* my_system */
#include <errno.h>		/* my_system */
#include <time.h>
#include <pwd.h>
#include <grp.h>
#include <string.h>
#include <ctype.h>
#if defined(AIX) || defined(_AIX) || defined(__aix__)
   #include <sys/select.h>
#endif

#include "fsusage.h"
#include "mountlist.h"

/* From dialog.h (not wanting to include it as
   it requires including a lot of other files, too) */
int message (int error, char *header, char *text, ...);

#include "mad.h"
#include "regex.h"
#include "util.h"
#include "global.h"
#include "profile.h"
#include "user.h"		/* expand_format */
typedef struct Panel Panel;
#include "tree.h"		/* tree_add_entry and tree_remove_entry for my_mkdir and my_rmdir */

static char rcsid [] = "$Id: util.c,v 1.16 1995/01/27 02:36:37 miguel Exp $";

char app_text [] = "Midnight-Commander";

int easy_patterns = 1;
int align_extensions = 1;
int tilde_trunc = 1;

#ifdef ultrix
char *strdup (char *s)
{
    char *t = malloc (strlen (s)+1);
    strcpy (t, s);
    return t;
}
#endif

int is_printable (int c)
{
    extern int eight_bit_clean;

    c &= 0xff;
    if (eight_bit_clean)
	return ((c >31 && c < 127) || c >= 160);
    else
	return (c > 31 && c < 127);
}

/* Returns the message dimensions (lines and columns) */
int msglen (char *text, int *lines)
{
    int max = 0;
    int line_len = 0;
    
    for (*lines = 1;*text; text++){
	if (*text == '\n'){
	    line_len = 0;
	    (*lines)++;
	} else {
	    line_len++;
	    if (line_len > max)
		max = line_len;
	}
    }
    return max;
}

char *trim (char *s, char *d, int len)
{
    int source_len = strlen (s);
    
    if (source_len > len){
	strcpy (d, s+(source_len-len));
	d [0] = '.';
	d [1] = '.';
	d [2] = '.';
    } else
	strcpy (d, s);
    return d;
}

char *name_trunc (char *txt, int trunc_len)
{
    static char x [MAXPATHLEN+MAXPATHLEN];
    int    txt_len;

    if (trunc_len > sizeof (x)-1){
	fprintf (stderr, "name_trunc: too big");
	trunc_len = sizeof (x)-1;
    }
    txt_len = strlen (txt);
    if (txt_len <= trunc_len)
	return txt;

    if (tilde_trunc){
	int y = trunc_len % 2;
	strncpy (x, txt, (trunc_len/2)+y);
	strncpy (x+(trunc_len/2)+y, txt+txt_len-(trunc_len/2), trunc_len/2);
	x [(trunc_len/2)+y] = '~';
    } else {
	strncpy (x, txt, trunc_len-1);
	x [trunc_len-1] = '>';
    }
    x [trunc_len] = 0;
    return x;
}

char *size_trunc (long int size)
{
    static char x [30];
    long int divisor = 1;
    char *xtra = "";
    
    if (size > 999999999L){
	divisor = 1024;
	xtra = "kb";
	if (size/divisor > 999999999L){
	    divisor = 1024*1024;
	    xtra = "Mb";
	}
    }
    sprintf (x, "%ld%s", (size/divisor), xtra);
    return x;
}

char *size_trunc_sep (long int size)
{
    static char x [60];
    int  count;
    char *p, *d, *y;

    p = y = size_trunc (size);
    p += strlen (p) - 1;
    d = x + sizeof (x) - 1;
    *d-- = 0;
    while (p >= y && isalpha (*p))
	*d-- = *p--;
    for (count = 0; p >= y; count++){
	if (count == 3){
	    *d-- = ',';
	    count = 0;
	}
	*d-- = *p--;
    }
    d++;
    if (*d == ',')
	d++;
    return d;
}

int is_exe (mode_t mode)
{
    if ((S_IXUSR & mode) || (S_IXGRP & mode) || (S_IXOTH & mode))
	return 1;
    return 0;
}

#define ismode(n,m) ((n & m) == m)

char *string_perm (mode_t mode_bits)
{
    static char mode [11];

    strcpy (mode, "----------");
    if (ismode (mode_bits, S_IFDIR)) mode [0] = 'd';
    if (ismode (mode_bits, S_IFSOCK)) mode [0] = 's';
    if (ismode (mode_bits, S_IXOTH)) mode [9] = 'x';
    if (ismode (mode_bits, S_IWOTH)) mode [8] = 'w';
    if (ismode (mode_bits, S_IROTH)) mode [7] = 'r';
    if (ismode (mode_bits, S_IXGRP)) mode [6] = 'x';
    if (ismode (mode_bits, S_IWGRP)) mode [5] = 'w';
    if (ismode (mode_bits, S_IRGRP)) mode [4] = 'r';
    if (ismode (mode_bits, S_IXUSR)) mode [3] = 'x';
    if (ismode (mode_bits, S_IWUSR)) mode [2] = 'w';
    if (ismode (mode_bits, S_IRUSR)) mode [1] = 'r';
    if (ismode (mode_bits, S_ISUID)) mode [3] = 's';
    if (ismode (mode_bits, S_ISGID)) mode [6] = 's';
    if (ismode (mode_bits, S_IFCHR)) mode [0] = 'c';
    if (ismode (mode_bits, S_IFBLK)) mode [0] = 'b';
    if (ismode (mode_bits, S_ISVTX)) mode [9] = 't';
    if (ismode (mode_bits, S_IFLNK)) mode [0] = 'l';
    if (ismode (mode_bits, S_IFIFO)) mode [0] = 's';
    return mode;
}

char *strip_home(char *dir)
{
    static char newdir [MAXPATHLEN];

    if (home_dir && !strncmp (dir, home_dir, strlen (home_dir))){
	newdir [0] = '~';
	strcpy (&newdir [1], &dir [strlen (home_dir)]);
	return newdir;
    } else
	return dir;
}

static char *maybe_start_group (char *d, int do_group, int *was_wildcard)
{
    if (!do_group)
	return d;
    if (*was_wildcard)
	return d;
    *was_wildcard = 1;
    *d++ = '\\';
    *d++ = '(';
    return d;
}

static char *maybe_end_group (char *d, int do_group, int *was_wildcard)
{
    if (!do_group)
	return d;
    if (!*was_wildcard)
	return d;
    *was_wildcard = 0;
    *d++ = '\\';
    *d++ = ')';
    return d;
}

/* If shell patterns are on converts a shell pattern to a regular
   expression. Called by regexp_match and mask_rename. */
char *convert_pattern (char *pattern, int match_type, int do_group)
{
    char *s, *d;
    static char new_pattern [100];
    int was_wildcard = 0;

    if (easy_patterns){
	d = new_pattern;
	if (match_type == match_file)
	    *d++ = '^';
	for (s = pattern; *s; s++, d++){
	    switch (*s){
	    case '*':
		d = maybe_start_group (d, do_group, &was_wildcard);
		*d++ = '.';
		*d   = '*';
		break;
		
	    case '?':
		d = maybe_start_group (d, do_group, &was_wildcard);
		*d = '.';
		break;
		
	    case '.':
		d = maybe_end_group (d, do_group, &was_wildcard);
		*d++ = '\\';
		*d   = '.';
		break;

	    default:
		d = maybe_end_group (d, do_group, &was_wildcard);
		*d = *s;
		break;
	    }
	}
	d = maybe_end_group (d, do_group, &was_wildcard);
	if (match_type == match_file)
	    *d++ = '$';
	*d = 0;
	return new_pattern;
    } else
	return pattern;
}

/* 1 if string matches
   0 if string doesn't match
   -1 if error in pattern */

int regexp_match (char *pattern, char *string, int match_type)
{
    static regex_t r;
    static char *old_pattern = NULL;
    static int old_type;
    int    rval;

    if (!old_pattern || strcmp (old_pattern, pattern) || old_type != match_type){
	if (old_pattern){
	    regfree (&r);
	    free (old_pattern);
	}
	pattern = convert_pattern (pattern, match_type, 0);
	if (regcomp (&r, pattern, REG_EXTENDED|REG_NOSUB))
	    return -1;
	old_pattern = strdup (pattern);
	old_type = match_type;
    }
    rval = !regexec (&r, string, 0, NULL, 0);
    return rval;
}

char *extension (char *filename)
{
    char *d;

    if (!strlen (filename))
	return "";
    
    d = filename + strlen (filename) - 1;
    for (;d >= filename; d--){
	if (*d == '.')
	    return d+1;
    }
    return "";
}

/* This routine uses the fact that x is at most 14 chars or so */
char *split_extension (char *x, int pad)
{
    return x;

    /* Buggy code 
    if (!align_extensions)
	return x;

    if (strlen (x) >= pad)
	return x;
    
    if ((ext = extension (x)) == x || *ext == 0)
	return x;

    strcpy (xbuf, x);
    for (i = strlen (x); i < pad; i++)
	xbuf [i] = ' ';
    xbuf [pad] = 0;

    l = strlen (ext);
    for (i = 0; i < l; i++)
	xbuf [pad-i] = *(ext+l-i-1);
    for (i = xbuf + (ext - x); i < 
    return xbuf; */
}

#ifndef HAVE_MAD
void *xmalloc (int size, char *source)
{
    void *m = malloc (size);

    if (!m){
	fprintf (stderr, "memory exhausted in module (%s)\n", source);
	exit (1);
    }
    return m;
}
#endif /* HAVE_MAD */

int get_int (char *file, char *key, int def)
{
    return GetPrivateProfileInt (app_text, key, def, file);
}

int set_int (char *file, char *key, int value)
{
    char buffer [6];

    sprintf (buffer,  "%d", value);
    return WritePrivateProfileString (app_text, key, buffer, file);
}

int exist_file (char *name)
{
    return access (name, R_OK) == 0;
}

char *load_file (char *filename)
{
    FILE *data_file;
    struct stat s;
    char *data;
    long read_size;
    
    if (stat (filename, &s) != 0){
	return 0;
    }
    if ((data_file = fopen (filename, "r")) == NULL){
	return 0;
    }
    data = (char *) xmalloc (s.st_size+1, "util, load_file");
    read_size = fread (data, 1, s.st_size, data_file);
    data [read_size] = 0;
    fclose (data_file);

    if (read_size > 0)
	return data;
    else
	return 0;
}

#define UID_CACHE_SIZE 200
#define GID_CACHE_SIZE 30

typedef struct {
    int  index;
    char *string;
} int_cache;

int_cache uid_cache [UID_CACHE_SIZE];
int_cache gid_cache [GID_CACHE_SIZE];

void init_uid_gid_cache (void)
{
    int i;

    for (i = 0; i < UID_CACHE_SIZE; i++)
	uid_cache [i].string = 0;

    for (i = 0; i < GID_CACHE_SIZE; i++)
	 gid_cache [i].string = 0;
}

static char *i_cache_match (int id, int_cache *cache, int size)
{
    int i;

    for (i = 0; i < size; i++)
	if (cache [i].index == id)
	    return cache [i].string;
    return 0;
}

static void i_cache_add (int id, int_cache *cache, int size, char *text,
			 int *last)
{
    if (cache [*last].string)
	free (cache [*last].string);
    cache [*last].string = strdup (text);
    cache [*last].index = id;
    *last = ((*last)+1) % size;
}

char *get_owner (int uid)
{
    struct passwd *pwd;
    static char ibuf [8];
    char   *name;
    static uid_last;
    
    if ((name = i_cache_match (uid, uid_cache, UID_CACHE_SIZE)) != NULL)
	return name;
    
    pwd = getpwuid (uid);
    if (pwd){
	i_cache_add (uid, uid_cache, UID_CACHE_SIZE, pwd->pw_name, &uid_last);
	return pwd->pw_name;
    }
    else {
	sprintf (ibuf, "%d", uid);
	return ibuf;
    }
}

char *get_group (int gid)
{
    struct group *grp;
    static char gbuf [8];
    char *name;
    static int  gid_last;
    
    if ((name = i_cache_match (gid, gid_cache, GID_CACHE_SIZE)) != NULL)
	return name;
    
    grp = getgrgid (gid);
    if (grp){
	i_cache_add (gid, gid_cache, GID_CACHE_SIZE, grp->gr_name, &gid_last);
	return grp->gr_name;
    } else {
	sprintf (gbuf, "%d", gid);
	return gbuf;
    }
}

char *file_date (time_t when)
{
    static char timebuf [40];
    time_t current_time = time ((time_t) 0);

    strcpy (timebuf, ctime (&when));
    if (current_time > when + 6L * 30L * 24L * 60L * 60L /* Old. */
	|| current_time < when - 60L * 60L) /* In the future. */
    {
	/* The file is fairly old or in the future.
	   POSIX says the cutoff is 6 months old;
	   approximate this by 6*30 days.
	   Allow a 1 hour slop factor for what is considered "the future",
	   to allow for NFS server/client clock disagreement.
	   Show the year instead of the time of day.  */
	strcpy (timebuf + 11, timebuf + 19);
    }
    timebuf[16] = 0;
    return &timebuf [4];
}

/* Like file_date, but packs the data to fit in 10 columns */
char *file_date_pck (time_t when)
{
    /* FIXME: Should return only 10 chars, not 14 */
    return file_date (when);
}

char *extract_line (char *s, char *top)
{
    static char tmp_line [500];
    char *t = tmp_line;
    
    while (*s && *s != '\n' && (t - tmp_line) < sizeof (tmp_line)-1 && s < top)
	*t++ = *s++;
    *t = 0;
    return tmp_line;
}

char *icase_search (char *text, char *data)
{
    char *d = text;
    char *e = data;
    
    for (;*e; e++){
	if ((*d | 0x20) == (*e | 0x20))
	    d++;
	else
	    d = text;
	if (!*d)
	    return e+1;
    }
    return 0;
}

#ifndef HAVE_BASENAME
char *basename (char *s)
{
    char  *where;
    return ((where = strrchr (s, '/')))? where + 1 : s;
}
#endif

char *get_full_name (char *dir, char *file)
{
    char *d = malloc (strlen (dir) + strlen (file) + 2);

    strcpy (d, dir);
    if (dir [0] != '/' || dir [1] != 0)
	strcat (d, "/");
    file = basename (file);
    strcat (d, file);
    return d;
}

/* Pipes are guaranteed to be able to hold at least 4096 bytes */
/* More than that would be unportable */
#define MAX_PIPE_SIZE 4096

static int error_pipe[2];	/* File descriptors of error pipe */
static int old_error;		/* File descriptor of old standard error */

/* Creates a pipe to hold standard error for a later analysis. */
/* The pipe can hold 4096 bytes. Make sure no more is written */
/* or a deadlock might occur. */
void open_error_pipe (void)
{
    if (pipe (error_pipe) < 0){
	message (0, " Warning ", " Pipe failed ");
    }
    old_error = dup (2);
    if(old_error < 0 || close(2) || dup (error_pipe[1]) != 2){
	message (0, " Warning ", " Dup failed ");
	close (error_pipe[0]);
	close (error_pipe[1]);
    }
    close (error_pipe[1]);
}

/* Checks for messages in the error pipe
 * and displays an warning box if needed
 */
void check_error_pipe (void)
{
    char error[MAX_PIPE_SIZE];
    int len = 0;
    if (old_error >= 0){
	while (len < MAX_PIPE_SIZE)
	{
            struct fd_set select_set;
            struct timeval timeout;
            FD_ZERO (&select_set);
            FD_SET (error_pipe[0], &select_set);
            timeout.tv_sec = 0;
            timeout.tv_usec = 0;
            select (FD_SETSIZE, &select_set, 0, 0, &timeout);
            if (!FD_ISSET (0, &select_set)) break;
	    read (error_pipe[0], error + len, 1);
	    len ++;
	}
	error[len] = 0;
	close (error_pipe[0]);
    }
    if (len > 0)
        message (0, " Warning ", error);
}

/* Checks for messages in the error pipe,
 * closes the pipe and displays an error box if needed
 */
void close_error_pipe (int error, char *text)
{
    char *title;
    char msg[MAX_PIPE_SIZE];
    int len = 0;

    if (error)
	title = " Error ";
    else
	title = " Warning ";
    if (old_error >= 0){
	close (2);
	dup (old_error);
	close (old_error);
	len = read (error_pipe[0], msg, MAX_PIPE_SIZE);
	msg[len] = 0;
	close (error_pipe[0]);
    }
    if (error < 0)
	return;		/* Just ignore error message */
    if (text == NULL){
	if (len == 0) return;	/* Nothing to show */

	/* Show message from pipe */
	message (error, title, msg);
    } else {
	/* Show given text and possible message from pipe */
	message (error, title, " %s \n %s ", text, msg);
    }
}

void my_putenv (char *name, char *data)
{
    char *full;

    full = xmalloc (strlen (name) + strlen (data) + 2, "util, my_putenv");
    sprintf (full, "%s=%s", name, data);
    putenv (full);
    /* WARNING: NEVER FREE THE full VARIABLE!!!!!!!!!!!!!!!!!!!!!!!! */
    /* It is used by putenv. Freeing it will corrupt the environment */
}

#if 0
static void my_putenv_expand (char *name, char macro_code)
{
    char *data;

    data = expand_format (macro_code);
    my_putenv (name, data);
    free (data);
}

/* Puts some status information in to the environment so that
   processes to be executed can access it. */
static void prepare_environment (void)
{
    my_putenv_expand ("MC_CURRENT_DIR", 'd');
    my_putenv_expand ("MC_OTHER_DIR", 'D');
    my_putenv_expand ("MC_CURRENT_FILE", 'f');
    my_putenv_expand ("MC_OTHER_FILE", 'F');
    my_putenv_expand ("MC_CURRENT_TAGGED", 't');
    my_putenv_expand ("MC_OTHER_TAGGED", 'T');
    /* MC_CONTROL_FILE has been added to environment on startup */
}
#endif

struct sigaction startup_handler;

int my_system (int as_shell_command, const char *shell, const char *command)
{
    struct sigaction ignore, save_intr, save_quit, save_stop;
    sigset_t child_mask, save_mask; 
    pid_t pid;
    int status = 0;

    ignore.sa_handler = SIG_IGN;
    sigemptyset (&ignore.sa_mask);
    ignore.sa_flags = 0;
    
    sigaction (SIGINT, &ignore, &save_intr);    
    sigaction (SIGQUIT, &ignore, &save_quit);

    /* Restore the original SIGTSTP handler, we don't want ncurses' */
    /* handler messing the screen after the SIGCONT */
    sigaction (SIGTSTP, &startup_handler, &save_stop);

    sigemptyset (&child_mask);
    sigaddset (&child_mask, SIGCHLD);
    sigprocmask (SIG_BLOCK, &child_mask, &save_mask);

    if ((pid = fork ()) < 0){
	fprintf (stderr, "\n\nfork () = -1\n");
	return -1;
    }
    if (pid == 0){
	sigaction (SIGINT,  &save_intr, NULL);
	sigaction (SIGQUIT, &save_quit, NULL);
	sigprocmask (SIG_SETMASK, &save_mask, NULL);

	#if 0
	prepare_environment ();
	#endif
	
	if (as_shell_command)
	    execl (shell, shell, "-c", command, (char *) 0);
	else
	    execlp (shell, shell, command, (char *) 0);

	_exit (127);		/* Exec error */
    } else {
	while (waitpid (pid, &status, 0) < 0)
	    if (errno != EINTR){
		status = -1;
		break;
	    }
    }
    sigprocmask (SIG_SETMASK, &save_mask, NULL);
    sigaction (SIGINT,  &save_intr, NULL);
    sigaction (SIGQUIT, &save_quit, NULL);
    sigaction (SIGTSTP, &save_stop, NULL);

    return status;
}

/* Since ncurses uses a handler that automatically refreshes the */
/* screen after a SIGCONT, and we don't want this behavior when */
/* spawning a child, we save the original handler here */
void save_stop_handler (void)
{
    sigaction (SIGTSTP, NULL, &startup_handler);
}

char *unix_error_string (int error_num)
{
    static char buffer [256];
    char *error_msg;
	
#ifdef HAVE_STRERROR
    error_msg = strerror (error_num);
#else
    extern int sys_nerr;
    extern char *sys_errlist [];
    if ((0 <= error_num) && (error_num < sys_nerr))
	error_msg = sys_errlist[error_num];
    else
	error_msg = "strange errno";
#endif
    sprintf (buffer, "%s (%d)", error_msg, error_num);
    return buffer;
}

/* Returns a newly allocated string */
char *tilde_expand (char *directory)
{
    struct passwd *passwd;
    char *p;
    char *name;
    int  len;
    
    if (*directory != '~')
	return strdup (directory);

    directory++;
    
    p = strchr (directory, '/');
    
    /* d = "~" or d = "~/" */
    if (!(*directory) || (*directory == '/')){
	passwd = getpwuid (geteuid ());
	p = (*directory == '/') ? directory+1 : "";
    } else {
	if (!p){
	    p = "";
	    passwd = getpwnam (directory);
	} else {
	    name = xmalloc (p - directory + 1, "tilde_expand");
	    strncpy (name, directory, p - directory);
	    name [p - directory] = 0;
	    passwd = getpwnam (name);
	    free (name);
	}
    }

    /* If we can't figure the user name, return home_dir */
    if (!passwd)
	return strdup (home_dir);

    len = strlen (passwd->pw_dir) + strlen (p) + 2;
    directory = xmalloc (len, "tilde_expand");
    strcpy (directory, passwd->pw_dir);
    strcat (directory, "/");
    strcat (directory, p);
    return directory;
}

char *copy_strings (char *first,...)
{
    va_list ap;
    int len;
    char *data, *result;

    if (!first)
	return 0;
    
    len = strlen (first);
    va_start (ap, first);

    while ((data = va_arg (ap, char *))!=0)
	len += strlen (data);

    len++;

    result = xmalloc (len, "copy_strings");
    va_end (ap);
    va_start (ap, first);
    strcpy (result, first);
    while ((data = va_arg (ap, char *)) != 0)
	strcat (result, data);
    va_end (ap);

    return result;
}
	
/* A function to check if we're idle.
   Currently checks only for key presses.
   We could also check the mouse. */
int is_idle (void)
{
    /* Check for incoming key presses     *
     * If there are any we say we're busy */

    struct fd_set select_set;
    struct timeval timeout;
    FD_ZERO (&select_set);
    FD_SET (0, &select_set);
    timeout.tv_sec = 0;
    timeout.tv_usec = 0;
    select (FD_SETSIZE, &select_set, 0, 0, &timeout);
    return ! FD_ISSET (0, &select_set);
}

long blocks2kilos (int blocks, int bsize)
{
    if (bsize > 1024){
	return blocks * (bsize / 1024);
    } else if (bsize < 1024){
	return blocks / (1024 /bsize);
    } else
	return blocks;
}

static struct mount_entry *mount_list;

void init_my_statfs (void)
{
#ifndef NO_INFOMOUNT
    mount_list = read_filesystem_list (1, 1);
#endif
}

void my_statfs (struct my_statfs *myfs_stats, char *path)
{
    int i, len = 0;

#ifndef NO_INFOMOUNT
    struct mount_entry *entry = NULL;
    struct mount_entry *temp = mount_list;
    struct fs_usage fs_use;

    while (temp){
	i = strlen (temp->me_mountdir);
	if (i > len && strncmp (path, temp->me_mountdir, i) == 0){
	    len = i;
	    entry = temp;
	}
	temp = temp->me_next;
    }

    if (entry){
	get_fs_usage (entry->me_mountdir, &fs_use);

	myfs_stats->type = entry->me_dev;
	myfs_stats->typename = entry->me_type;
	myfs_stats->mpoint = entry->me_mountdir;
	myfs_stats->device = entry->me_devname;
	myfs_stats->avail = getuid () ? fs_use.fsu_bavail/2 : fs_use.fsu_bfree/2;
	myfs_stats->total = fs_use.fsu_blocks/2;
	myfs_stats->nfree = fs_use.fsu_ffree;
	myfs_stats->nodes = fs_use.fsu_files;
    } else
#endif
    {
	myfs_stats->type = 0;
	myfs_stats->mpoint = "unknown";
	myfs_stats->device = "unknown";
	myfs_stats->avail = 0;
	myfs_stats->total = 0;
	myfs_stats->nfree = 0;
	myfs_stats->nodes = 0;
    }
}

char *skip_separators (char *s)
{
    for (;*s; s++)
	if (*s != ' ' && *s != '\t' && *s != ',')
	    break;
    return s;
}

char *skip_numbers (char *s)
{
    for (;*s; s++)
	if (!isdigit (*s))
	    break;
    return s;
}

/* Remove all control sequences from the argument string.  We define
 * "control sequence", in a sort of pidgin BNF, as follows:
 *
 * control-seq = Esc non-'['
 *	       | Esc '[' (0 or more digits or ';' or '?') (any other char)
 *
 * This scheme works for all the terminals described in my termcap /
 * terminfo databases, except the Hewlett-Packard 70092 and some Wyse
 * terminals.  If I hear from a single person who uses such a terminal
 * with MC, I'll be glad to add support for it.  (Dugan)
 */

char *strip_ctrl_codes (char *s)
{
    int i;  /* Current length of the string's correct (stripped) prefix */
    int j;  /* Number of control characters we have skipped so far */

    if (!s)
	return 0;
    
    for (i = 0, j = 0; s [i+j]; ++i)
	if (s [i+j] != ESC_CHAR){
	    if (j)
		s [i] = s [i+j];
	} else {
	    ++j;
	    if (s [i+j++] == '[')
		while (strchr ("0123456789;?", s [i+j++]))
		    /* Skip the control sequence's arguments */ ;
	    --i;
	}
    s[i] = 0;
    return s;
}

#ifndef HAVE_STRCASECMP
/* At least one version of HP/UX lacks this */
/* Assumes ASCII encoding */
int strcasecmp (char *s, char *d)
{
    register char result;

    while (1){
	if (result = (0x20 | *s) - (0x20 | *d))
	    break;
	if (!*s)
	    return 0;
	s++;
	d++;
    }
    return result;
}
#endif /* HAVE_STRCASECMP */

/* getwd is better than getcwd, the later uses a popen ("pwd"); */
char *get_current_wd (char *buffer, int size)
{
    char *p;
    
#ifdef HAVE_GETWD
    p = (char *) getwd (buffer);
#else
    p = getcwd (buffer, size);
#endif
    return p;
}

static char *get_absolute_name (char *file)
{
    char dir [MAXPATHLEN];

    if (file [0] == '/')
	return strdup (file);
    get_current_wd (dir, MAXPATHLEN);
    return get_full_name (dir, file);
}

int my_mkdir (char *s, mode_t mode)
{
    int result;

    result = mkdir (s, mode);
    if (result == 0){
	s = get_absolute_name (s);
	tree_add_entry (s);
	free (s);
    }
    return result;
}

int my_rmdir (char *s)
{
    int result;

    result = rmdir (s);
    if (result == 0){
	s = get_absolute_name (s);
	tree_remove_entry (s);
	free (s);
    }
    return result;
}
