------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ A G G R                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.7 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Errout;   use Errout;
with Expander; use Expander;
with Features; use Features;
with Itypes;   use Itypes;
with Nmake;    use Nmake;
with Nlists;   use Nlists;
with Opt;      use Opt;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Ch3;  use Sem_Ch3;
with Sem_Ch5;  use Sem_Ch5;
with Sem_Eval; use Sem_Eval;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sem_Type; use Sem_Type;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stand;    use Stand;
with Tbuild;   use Tbuild;
with Uintp;    use Uintp;

with System.Parameters;

package body Exp_Aggr is

   procedure Convert_To_Assignments (N : Node_Id; Typ : Entity_Id);
   --  If the aggregate appears in a context with no target object, or if
   --  it contains controllwed component, the aggregate is expanded into a
   --  definition for a temporary, followed by a series of assignments to
   --  individual components.

   procedure Expand_Record_Aggregate (N : Node_Id);
   --  This is the top level procedure for record aggregate expansion.
   --
   --    N is the record aggregate node.
   --
   --  Expansion for record aggregates needs only expand aggregates for
   --  tagged record types. Specifically Expand_Record_Aggregate adds the
   --  Tag field in front of the Component_Association list that was created
   --  during resolution by Resolve_Record_Aggregate. For derived tagged
   --  records it also needs to add the _Parent field and recursively regroup
   --  the components belonging to _Parent into a nested record aggregate.
   --
   --  The remaining required record aggregate normalisations are done in
   --  Resolve_Record_Aggregate since that has to be done there anyway
   --  to perform all the semantic checks.

   procedure Expand_Array_Aggregate (N : Node_Id);
   --  This procedure is the top-level procedure to perform array
   --  aggregate expansion.
   --
   --  Two algorithms are used for expansion.
   --
   --  The first one represented by the function Rewrite_Array_Aggregate is
   --  called only in very special cases : the array type should be constrained
   --  the indices of the array should be static, in the case of a named
   --  aggregate the choices should be all static and when there is an Others
   --  statements the number of values which are represented by Others should
   --  be less then a threshold fixed to 1024 values.
   --
   --  The original tree of the aggregate is replaced by an other aggregate
   --  which contains only positional associations :
   --
   --    (1, 2, 3, 4, 5)        => (1, 2, 3, 4, 5);
   --    (1, 2, 3, others => 4) => (1, 2, 3, 4, 4);
   --    (2 .. 3 | 5 => 1, 1 | 4 => 2)  => (2, 1, 1, 2, 1);
   --    (2 .. 3 | 5 => 1, others => 2) => (2, 1, 1, 2, 1);
   --
   --  The second algorithm used in all other cases builds an expression
   --  actions node containing the declaration of a temporary array whose type
   --  is the type of the aggregate and containing the necessary loops or/and
   --  assignments to fill the temporary with the values specified in the
   --  aggregate. The expression of the node expression actions is the
   --  temporary filled previously. The declaration of the temporary is not
   --  necessary in all cases but it's simpler to create one each time.
   --  the original node is replaced by this expression actions node :
   --
   --    (M .. N => Val) => {<actions>
   --                           Tmp : Array_Type;
   --                           for J in M .. N loop
   --                              Tmp (J) := Val;
   --                           end loop;
   --                        <expression>
   --                           Tmp;}
   --
   --  This algorithm is used specially when the bound of the indices of the
   --  array are dynamic or when the array is an unconstrained array.
   --  When the semantic analysis has been able to determine that a run time
   --  Constraint Error will be raised then the aggregate is modified by :
   --  in the first algorithm, we build a node expression actions containing
   --  raise constraint_error for its actions and a correct aggregate (needed
   --  by the back end) whose values are arbitrary (in fact the first
   --  expression of the original aggregate is choosen) fills the expression
   --  field.
   --
   --    (1, 2, 4, 5) => {<actions> raise Constraint_Error;
   --                     <expression> (1, 1, 1, 1, 1);}
   --
   --  in the second algorithm, we build a node expression actions containing
   --  a declaration of a temporary and the statememt raise constraint error
   --  for its actions and the temporary for its expression.
   --
   --    (1 .. 2 => 1, 3 .. 6 => 2) => {<actions>
   --                                       Tmp : Array_Type;
   --                                       raise Constraint_Error;
   --                                    <expression>
   --                                       Tmp;}
   --
   --  The expansion of the aggregate is called only if we are in a code
   --  generating phase.

   function Raise_CE (N : Node_Id) return Boolean;
   --  Checks whether the aggregate N contains a raise Constraint_Error.
   --  Returns True if yes.

   function Static_Processing_Possible (N : Node_Id) return Boolean;
   --  This function checks if it possible to build a fully positional array
   --  aggregate at compile time. If this is possible True is returned.
   --
   --  Static processing for the whole array aggregate is possible only if
   --  it is possible for all of its sub-aggregates. Static processing for
   --  a sub-aggregate is possible if:
   --
   --    1. S has an others choice and the bounds of the corresponding
   --       index range are static;
   --
   --    2. S is fully positional and has no others choice;
   --
   --    3. S has only static named components and no others choice.
   --
   --    4. The index component for S is not an enumeration type with
   --       non-standard representation.
   --
   --  Note that if the overall number of positional elements that need to
   --  be *added* to the new aggregate (because of an others choice or a range
   --  choice) is bigger than Max_Add, we do not consider static processing
   --  as being possible and return False.

   function Rewrite_Array_Aggregate
     (N     : Node_Id;
      Index : Node_Id)
      return  Node_Id;
   --  This function is called only if static processing is deemed possible
   --  (Static_Processing_Possible returns True). If this is the case the
   --  routine recursively constructs a new N_Aggregate node containing
   --  only positional components.
   --
   --    N is an array aggregate or sub-aggregate to be rewritten in
   --    positional form.
   --
   --    Index is the array index corresponding to the array (sub)aggregate
   --
   --  The function returns the transformed N_Aggregate node
   --
   --  Examples of aggregate rewrites are:
   --
   --  1. Positional aggregate
   --        A : array (1..6) of integer := (1, 2, 3, 4, others => 5);
   --     => A : array (1..6) of integer := (1, 2, 3, 4, 5, 5);
   --
   --  2. Named aggregate :
   --        A : array (1..6) of integer := (2..3|6 => 1, 4 => 4, others => 0);
   --     => A : array (1..6) of integer := (0, 1, 1, 0, 4, 0, 1);

   function Build_Loop
     (N    : Node_Id;
      Ind  : Entity_Id;
      L    : List_Id)
      return List_Id;
   --  This procedure builds a list which contains the loops and assignments
   --  that are needed for the expansion of the aggregate. It's part
   --  of the second algorithm of expansion of the aggregate.
   --
   --  First, we look if there's an Others statement in the aggregate which
   --  is represented by the node N. If it's the case, we build the code
   --  corresponding to :
   --
   --     for J in Index_Low_Bound .. Index_High_Bound loop
   --
   --  If we deal with a multi dimensional aggregate, we call recursively
   --  Build_Loop on the expression represented by Others. Else, we build
   --  the code corresponding to the assignment of the expression :
   --
   --     Tmp (List_Of_Indices, L) := Expression;
   --
   --  Then, we deal with the other named associations if they exist :
   --  if the choice of the named association is a range choice then we
   --  build a loop corresponding to the code :
   --
   --     for J in Choice_Low_Bound .. Choice_High_Bound loop
   --
   --  If the aggregate is multi dimensional we call recursively Build_Loop
   --  else we create the code correponding to the assignment :
   --
   --     Tmp (List_Of_Indices_Known, I) := Choice_Expression;
   --
   --  If the choice is a simple choice : if the aggregate is a multi
   --  dimensionla aggregtae we call recursively Build_loop else we create
   --  the code corresponding to the assignment :
   --
   --     Tmp (List_Of_Indices_Know, Choice) := Choice_Expression;
   --
   --  If the aggregate contains positional associations instead of named
   --  associations, we need to generate two sort of code considering the
   --  case of the index Ind corresponding to the aggregate N is an
   --  enumeration type or not. It this is the case, the code created is:
   --
   --     X : Index := Index'First;
   --
   --  For each positional association, an assignement is created and the
   --  value of X is updated to the next enumeration type in the range
   --  Index'first .. Index'last. The code created is:
   --
   --     X := X'succ;
   --     Tmp (List_Of_Indices_Known, X) := Positional;
   --
   --  If the index is not an enumeration type then the code is simpler:
   --
   --     Tmp (List_Of_Indices_Known, Index'First+Position) := Positional;
   --
   --  where Position represents the position (- 1) of the positional
   --  association in the aggregate (the position of the first positional
   --  is zero). the value of Index'first + Position is computed when
   --  analyze is called on this subtree.
   --
   --  The N node correspond to ther aggregate or subaggregate that needs
   --  to be expanded, Ind corresponds to the index in the array of this
   --  subaggregate and L corresponds to the list of known identifiers
   --  at this level of the aggregate : this is necessary for the case
   --  of created nested loops to create the correct list of indices in
   --  the built assignments. In the description given here, it corresponds
   --  to List_Of_Indices_Known for the code created.

   function Fill_Index (N : Node_Id; Ind : Entity_Id) return List_Id;
   --  This function is used only in case of an unconstrained array type
   --  for the aggregate. In this precise case we need to create a temporary
   --  whose type is the correct constrained type : So the purpose of this
   --  function is to create the code necessary to constrained the array by
   --  generating the range needed for each index knowing the number of
   --  associations existing in the aggregate. We remind the reader that in
   --  this precise case there is no Others statement possible.
   --
   --  First we deal with trhe case of named associations by looking for
   --  the low and high bound of the choices existing in the aggregate and
   --  we generate the code that will be added to the list of indices of
   --  the array type of the temporary:
   --
   --     Choice_Low_Bound .. Choice_High_Bound
   --
   --  In the case where the choices are not static then we are sure that
   --  there's only one association so we just copy its bounds.
   --  For positional associations the bounds are compute by taking
   --  Index'first and calculating the numebr of associations. The code
   --  created is:
   --
   --     Index_Low_Bound .. Index_Low_Bound + Nb_positional - 1
   --
   --  If the type of the indexx is an enumeration type the code created
   --  is different : we need to know which enumeration literal will be
   --  the high bound so we create the code:
   --
   --     Index_Low_Bound ..
   --       Enum'Val (Enum'Pos (Index_Low_Bound) + Nb_Positional - 1)
   --
   --  The N node represents the aggregate examined and Ind the index
   --  corresponding to this N aggregate.
   --
   --  This function is recursive for multi dimensional aggregate.

   --  The following global variable holds the temporary for the array
   --  aggregate being built. Made global for use in Expand_Array_Aggregate
   --  and in Build_Loop.
   --  Should Build_Loop be local to Expand_Array_Aggregate ???

   Tmp          : Entity_Id;

   ------------------------
   -- Expand_N_Aggregate --
   ------------------------

   procedure Expand_N_Aggregate (N : Node_Id) is
   begin
      if Is_Record_Type (Etype (N)) then
         Expand_Record_Aggregate (N);
      else
         Expand_Array_Aggregate (N);
      end if;
   end Expand_N_Aggregate;

   ----------------------------
   -- Convert_To_Assignments --
   ----------------------------

   procedure Convert_To_Assignments (N : Node_Id; Typ : Entity_Id) is
      Actions : List_Id := New_List;
      Comp    : Node_Id;
      Loc     : constant Source_Ptr := Sloc (N);
      Temp    : constant Entity_Id :=
                  Make_Defining_Identifier (Loc, New_Internal_Name ('A'));
      Instr   : Node_Id;

   begin

      if Is_Controlled (Typ) then
         Unimplemented (N, "aggregate of a controlled type");

      elsif Has_Controlled (Typ) then
         Unimplemented (N, "aggregate with controlled components");
      end if;

      Instr :=
        Make_Object_Declaration (Loc,
          Defining_Identifier => Temp,
          Object_Definition => New_Occurrence_Of (Typ, Loc));

      Set_No_Default_Init (Instr);
      Append_To (Actions, Instr);

      Comp := First (Component_Associations (N));

      while Present (Comp) loop
         Instr :=
           Make_Assignment_Statement (Loc,
             Name =>
               Make_Selected_Component (Loc,
                 Prefix => New_Occurrence_Of (Temp, Loc),
                  Selector_Name =>
                    New_Occurrence_Of
                      (Entity (First (Choices (Comp))), Loc)),
             Expression => Expression (Comp));

         Set_Assignment_OK (Name (Instr));
         Append_To (Actions, Instr);
         Comp := Next (Comp);
      end loop;

      --  if the type is tagged, the tag needs to be initialized

      if Is_Tagged_Type (Typ) then

         Instr :=
           Make_Assignment_Statement (Loc,
             Name =>
               Make_Selected_Component (Loc,
                 Prefix => New_Occurrence_Of (Temp, Loc),
                 Selector_Name =>
                   New_Reference_To (Tag_Component (Base_Type (Typ)), Loc)),

             Expression =>
               Make_Unchecked_Type_Conversion (Loc,
                 Subtype_Mark => New_Reference_To (RTE (RE_Tag), Loc),
                 Expression =>
                   New_Reference_To (Access_Disp_Table (Base_Type (Typ)),
                     Loc)));

         Set_Assignment_OK (Name (Instr));
         Append_To (Actions, Instr);
      end if;

      Rewrite_Substitute_Tree (N,
        Make_Expression_Actions (Loc,
          Actions    => Actions,
          Expression => New_Occurrence_Of (Temp, Loc)));
      Analyze (N);
   end Convert_To_Assignments;

   ----------------------------------
   -- Expand_N_Extension_Aggregate --
   ----------------------------------

   --  If the ancestor part is an expression, add a component association for
   --  the parent field. If the type of the ancestor part is not the direct
   --  parent of the expected type,  build recursively the needed ancestors.
   --  If the ancestor part is a subtype_mark, replace aggregate with a decla-
   --  ration for a temporary of the expected type, followed by individual
   --  assignments to the given components.

   procedure Expand_N_Extension_Aggregate (N : Node_Id) is
      A   : constant Node_Id    := Ancestor_Part (N);
      A_T : constant Entity_Id  := Etype (A);
      Loc : constant Source_Ptr := Sloc  (N);
      Typ : constant Entity_Id  := Etype (N);

      Comp         : Node_Id;
      First_Comp   : Node_Id;
      Parent_Assoc : Node_Id;
      Parent_Comps : List_Id;
      Parent_Def   : Node_Id;
      Parent_Ext   : Node_Id;
      Record_Def   : Node_Id;

   begin
      if Is_Entity_Name (A)
        and then Is_Type (Entity (A))
      then
         Convert_To_Assignments (N, Typ);

      else
         --  Find parent component in record definition, and add the
         --  corresponding component association to aggregate.

         Record_Def := Type_Definition (Parent (Base_Type (Typ)));

         Parent_Def := First (Component_Items (Component_List
           (Record_Extension_Part (Record_Def))));

         while Present (Parent_Def)
           and then Chars (Defining_Identifier (Parent_Def)) /= Name_uParent
         loop
            Parent_Def := Next (Parent_Def);
         end loop;

         if Etype (Base_Type (Typ)) = Base_Type (A_T) then

            --  ancestor expression is direct ancestor of expected type.

            Parent_Ext := A;

         else
            --  need to build intermediate ancestors. The components of the
            --  expected type that are inherited constitute an inner extension
            --  aggregate for the immediate parent. The components appear in
            --  order of inheritance in the association list, so we can skip
            --  forward over all inherited components.

            First_Comp := First (Component_Associations (N));
            Parent_Comps := New_List;

            while Scope (Entity (First (Choices (First_Comp)))) /= Typ loop
               Comp := First_Comp;
               First_Comp := Next (First_Comp);
               Remove (Comp);
               Append (Comp, Parent_Comps);
            end loop;

            Parent_Ext := Make_Extension_Aggregate (Loc,
              Ancestor_Part => A,
              Component_Associations => Parent_Comps);

            Set_Etype (Parent_Ext, Etype (Base_Type (Typ)));
         end if;

         Parent_Assoc :=
           Make_Component_Association (Loc,
             Choices => New_List
               (New_Occurrence_Of (Defining_Identifier (Parent_Def),  Loc)),
             Expression => Parent_Ext);

         --  The parent aggregate may mention a more remote ancestor.

         if Parent_Ext /= A then
            Expand_N_Extension_Aggregate (Parent_Ext);
         end if;

         Prepend (Parent_Assoc, To => Component_Associations (N));

         --  If the context is an assignment or an object declaration, the
         --  target of the aggregate has the correct tag and we have completed
         --  the construction of the aggregate by adding the association for
         --  the parent. If the context is an expression or a call, we need
         --  to create a temporary and expand the aggregate into component
         --  assignments, as when the ancestor part is a type mark. This
         --  expansion is always correct, but much more inefficient than the
         --  aggregate form, which is worth preserving when possible. Note in
         --  particular that an inner aggregate is not expanded.

         --  If the expected type is controlled, or has a controlled component,
         --  then we perform the expansion as well, to insure that chaining
         --  an finalization actions are done correctly.

         if (Nkind (Parent (N)) /= N_Object_Declaration
           and then Nkind (Parent (N)) /= N_Assignment_Statement
           and then Nkind (Parent (N)) /= N_Component_Association)
           or else Is_Controlled  (Typ)
           or else Has_Controlled (Typ)
         then
            Convert_To_Assignments (N, Typ);
         end if;
      end if;
   end Expand_N_Extension_Aggregate;

   -----------------------------
   -- Expand_Record_Aggregate --
   -----------------------------

   procedure Expand_Record_Aggregate (N : Node_Id) is
      Typ : constant Entity_Id  := Etype (N);
      Loc : constant Source_Ptr := Sloc  (N);

      Tag_Name  : Node_Id;
      Tag_Value : Node_Id;
      Conv_Node : Node_Id;

      Choice_List     : List_Id;
      Component_Assoc : Node_Id;

   begin

      if Is_Tagged_Type (Typ) then
         if Is_Derived_Type (Typ) then

            --  Convert the aggregate into an extension aggregate by collecting
            --  inherited components into an ancestor part for the immediate
            --  parent of the expected type.

            declare
               Comp         : Node_Id;
               First_Comp   : Node_Id;
               New_N        : Node_Id;
               Parent_Comps : List_Id;
               Parent_Ext   : Node_Id;

            begin
               First_Comp := First (Component_Associations (N));
               Parent_Comps := New_List;

               while Present (First_Comp)
                 and then Scope (Entity (First (Choices (First_Comp)))) /= Typ
               loop
                  Comp := First_Comp;
                  First_Comp := Next (First_Comp);
                  Remove (Comp);
                  Append (Comp, Parent_Comps);
               end loop;

               Parent_Ext := Make_Aggregate (Loc,
                 Component_Associations => Parent_Comps);
               Set_Etype (Parent_Ext, Etype (Base_Type (Typ)));
               Expand_Record_Aggregate (Parent_Ext);

               New_N := Make_Extension_Aggregate (Loc,
                 Ancestor_Part => Parent_Ext,
                 Component_Associations => Component_Associations (N));
               Set_Etype (New_N, Typ);
               Rewrite_Substitute_Tree (N, New_N);
               Expand_N_Extension_Aggregate (N);
            end;

         else
            --  If the expected type is tagged but not derived, add the
            --  association for the _Tag field in front of the aggregate,
            --  since the tag is the first component of a tagged type.

            Tag_Name  := New_Occurrence_Of (First_Component   (Typ), Loc);
            Tag_Value := New_Occurrence_Of (Access_Disp_Table (Typ), Loc);

            Conv_Node := Make_Unchecked_Type_Conversion
              (Loc,
               Subtype_Mark => New_Occurrence_Of (RTE (RE_Tag), Loc),
               Expression   => Tag_Value);
            Set_Etype (Conv_Node, RTE (RE_Tag));

            Choice_List := New_List;
            Append (Tag_Name, Choice_List);
            Component_Assoc :=
              Make_Component_Association (Loc,
                Choices    => Choice_List,
                Expression => Conv_Node);

            Prepend (Component_Assoc, To => Component_Associations (N));

            if Is_Controlled (Typ) then
               Convert_To_Assignments (N, Typ);
            end if;
         end if;

      elsif Has_Controlled (Typ)
        or else not Size_Known_At_Compile_Time (Typ)
      then
         Convert_To_Assignments (N, Typ);
      end if;
   end Expand_Record_Aggregate;

   ----------------------------
   -- Expand_Array_Aggregate --
   ----------------------------

   procedure Expand_Array_Aggregate (N : Node_Id) is
      Loc          : constant Source_Ptr := Sloc (N);
      Actions_List : List_Id := New_List;
      New_Aggr     : Node_Id := New_Node (N_Aggregate, Loc);
      Action       : Node_Id;
      Raise_C      : Boolean := False;   --  what is the purpose of this ???
      Typ          : constant Entity_Id := Etype (N);

   begin
      --  Aggregate consistency checks and bound evaluations should
      --  be performed here.???

      --  Also aggregate discrete_choices should be evaluated here
      --  and constraint checks should be emitted here???

      --  In particular all checks concerning N_Subtype_Indication should not
      --  be forgotten ???

      if Raise_CE (N) then
         Rewrite_Substitute_Tree (N, Make_Raise_Constraint_Error (Sloc (N)));
         Set_Raises_Constraint_Error (N);
         Set_Analyzed (N, True);
         Set_Etype (N, Typ);
         --  ??? Note that this Set_Etype will cause gigi to crash if Typ is
         --  unconstrained. Needs more work in that case.

         return;
      end if;

      if Is_Constrained (Typ) and then Static_Processing_Possible (N) then
         New_Aggr := Rewrite_Array_Aggregate (N, First_Index (Typ));
         Rewrite_Substitute_Tree (N, New_Aggr);
         Set_Etype (N, Typ);

      else
         --  If the array type of the aggregate is unconstrained then make
         --  an object declaration with a subtype indication whose index
         --  ranges will be filled by Fill_Index. The code created is:

         --    Array_Type (...);

         --  Else, create an object declaration whose code is:

         --    A : Array_Type;

         Tmp := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));

         if not Is_Constrained (Typ) then
            Append_To (Actions_List,
              Make_Object_Declaration (Loc,
                Defining_Identifier => Tmp,
                Object_Definition   =>
                  Make_Subtype_Indication (Loc,
                    Subtype_Mark => New_Occurrence_Of (Typ, Loc),
                    Constraint =>
                      Make_Index_Or_Discriminant_Constraint (Loc,
                        Constraints => Fill_Index (N, First_Index (Typ))))));

         elsif Is_Record_Type (Scope (Typ)) then

            --  The type is an implicit subtype which may refer discriminants
            --  we can't use it directly. We need to rebuild a brand new one
            --  in order to let discriminants be transformed in discriminals

            declare
               New_Constrs : constant List_Id :=
                 New_List_Copy_Tree (Constraints (Parent (First_Index (Typ))));
               C : Node_Id := First (New_Constrs);

            begin
               while Present (C) loop
                  Set_Analyzed (C, False);
                  if Nkind (C) = N_Range then
                     Set_Analyzed (Low_Bound (C), False);
                     Set_Analyzed (High_Bound (C), False);
                  end if;
                  C := Next (C);
               end loop;

               Append_To (Actions_List,
                 Make_Object_Declaration (Loc,
                   Defining_Identifier => Tmp,
                   Object_Definition   =>
                     Make_Subtype_Indication (Loc,
                       Subtype_Mark =>
                         New_Occurrence_Of (Base_Type (Typ), Loc),
                       Constraint =>
                         Make_Index_Or_Discriminant_Constraint (Loc,
                           Constraints => New_Constrs))));
            end;

         else
            Append_To (Actions_List,
              Make_Object_Declaration (Loc,
                Defining_Identifier => Tmp,
                Object_Definition   => New_Occurrence_Of (Typ, Loc)));
         end if;

         if Raise_C then
            Append_To (Actions_List,
              Make_Raise_Statement (Loc,
                New_Reference_To (Standard_Entity (S_Constraint_Error), Loc)));

         else
            Append_List
                 (Build_Loop (N, First_Index (Typ), New_List),
                  Actions_List);
         end if;

         Rewrite_Substitute_Tree (N,
           Make_Expression_Actions (Loc,
             Actions    => Actions_List,
             Expression => New_Reference_To (Tmp, Loc)));

         Analyze (N);
         Resolve (N, Etype (Tmp));
      end if;
   end Expand_Array_Aggregate;

   --------------
   -- Raise_CE --
   --------------

   function Raise_CE (N : Node_Id) return Boolean is

      function Raise_CE_SE (N : Node_Id; Index : Node_Id) return Boolean;
      --  This recursive function implements Raise_CE.

      function Raise_CE_SE (N : Node_Id; Index : Node_Id) return Boolean is
         Assoc  : Node_Id;
         Expr   : Node_Id;

      begin
         if Nkind (N) = N_Raise_Constraint_Error then
            return True;
         end if;

         --  Process component associations

         if Present (Component_Associations (N)) then
            Assoc := First (Component_Associations (N));
            while Present (Assoc) loop

               Expr := Expression (Assoc);

               if Nkind (Expr) = N_Raise_Constraint_Error then
                  return True;

               elsif Present (Next_Index (Index)) then
                  if Raise_CE_SE (Expr, Next_Index (Index)) then
                     return True;
                  end if;
               end if;

               Assoc := Next (Assoc);
            end loop;
         end if;

         --  Process positional components

         if Present (Expressions (N)) then
            Expr := First (Expressions (N));

            while Present (Expr) loop
               if Nkind (Expr) = N_Raise_Constraint_Error then
                  return True;

               elsif Present (Next_Index (Index)) then
                  if Raise_CE_SE (Expr, Next_Index (Index)) then
                     return True;
                  end if;
               end if;

               Expr := Next (Expr);
            end loop;
         end if;

         return False;
      end Raise_CE_SE;

   --  Start processing for Raise_CE

   begin
      return Raise_CE_SE (N, First_Index (Etype (N)));
   end Raise_CE;

   --------------------------------
   -- Static_Processing_Possible --
   --------------------------------

   function Static_Processing_Possible (N : Node_Id) return Boolean is

      Remaining_Size : Uint :=
                   UI_From_Int (System.Parameters.Max_Static_Aggregate_Size);
      --  Variable modified by side effect by Static_Processing_Possible_SE.
      --  Keeps track of the remaining number of positional components that
      --  can be added to the aggregate

      function Static_Processing_Possible_SE
        (N     : Node_Id;
         Index : Node_Id)
         return  Boolean;
      --  This recursive function implements Static_Processing_Possible.
      --  As this procedure is called, Remaining_Size is decremented to
      --  account for elements that are added.

      function Static_Processing_Possible_SE
        (N     : Node_Id;
         Index : Node_Id)
         return Boolean
      is
         Itype  : constant Entity_Id := Etype (Index);

         Assoc  : Node_Id;
         Choice : Node_Id;
         Expr   : Node_Id;

         Low  : Node_Id;
         High : Node_Id;

         Others_Present : Boolean;

         UI_Size  : Uint;
         --  overall number of aggregate elements to transform into positional

         Min_Low  : Uint;
         Max_High : Uint;
         --  keep track of the lowest and highest choices in a named aggregate

         Min_Max_Set : Boolean := False;
         --  is True after the above two variables receive their initial value

      begin
         --  No static processing if index subtype is an enumeration type
         --  with holes in it (this is a temporary expedient to get at least
         --  some of these aggregates correct)

         if Is_Enumeration_Type (Itype)
           and then Present (Enum_Pos_To_Rep (Base_Type (Itype)))
         then
            return False;
         end if;

         if Nkind (N) = N_Raise_Constraint_Error then
            return False;
         end if;

         --  Process component associations

         if Present (Component_Associations (N)) then
            Assoc := Last (Component_Associations (N));
            Others_Present :=
              (Nkind (First (Choices (Assoc))) = N_Others_Choice);

            Assoc := First (Component_Associations (N));
            while Present (Assoc) loop

               Choice := First (Choices (Assoc));
               while Present (Choice) loop

                  if Nkind (Choice) = N_Others_Choice then
                     exit;

                  else
                     Get_Index_Bounds (Choice, Low, High);
                  end if;

                  if not Is_OK_Static_Expression (Low)
                    or else not Is_OK_Static_Expression (High)
                  then
                     return False;

                  elsif not Min_Max_Set then
                     Min_Max_Set := True;
                     Min_Low  := Expr_Value (Low);
                     Max_High := Expr_Value (High);

                  else
                     Min_Low  := UI_Min (Min_Low,  Expr_Value (Low));
                     Max_High := UI_Max (Max_High, Expr_Value (High));
                  end if;

                  Choice := Next (Choice);
               end loop;

               Expr := Expression (Assoc);

               if Present (Next_Index (Index)) then
                  if not
                    Static_Processing_Possible_SE (Expr, Next_Index (Index))
                  then
                     return False;
                  end if;
               end if;

               Assoc := Next (Assoc);
            end loop;

            --  The aggregate takes its bounds from the corresponding
            --  index constraint if an others choice is present

            if Others_Present then

               --  For now, never consider an aggregate with an others
               --  choice as being static. This is a kludge to prevent
               --  giant static aggregates from being generated, and
               --  should be revisited later ???

               return False;

               Get_Index_Bounds (Index, Low, High);

               if Is_OK_Static_Expression (Low)
                 and then Is_OK_Static_Expression (High)
               then
                  UI_Size := 1 + (Expr_Value (High) - Expr_Value (Low));
               else
                  return False;
               end if;

            else
               UI_Size := 1 + (Max_High - Min_Low);
            end if;

            Remaining_Size := Remaining_Size - UI_Size;

            if Remaining_Size < 0 then
               return False;
            end if;
         end if;

         --  Process positional components

         if Present (Expressions (N)) then
            Expr := First (Expressions (N));

            while Present (Expr) loop
               if Present (Next_Index (Index)) then
                  if not
                    Static_Processing_Possible_SE (Expr, Next_Index (Index))
                  then
                     return False;
                  end if;
               end if;

               Expr := Next (Expr);
            end loop;
         end if;

         return True;
      end Static_Processing_Possible_SE;

   --  Start processing for Static_Processing_Possible

   begin
      return Static_Processing_Possible_SE (N, First_Index (Etype (N)));
   end Static_Processing_Possible;

   -----------------------------
   -- Rewrite_Array_Aggregate --
   -----------------------------

   function Rewrite_Array_Aggregate
     (N     : Node_Id;
      Index : Node_Id)
      return Node_Id
   is
      Assoc  : Node_Id;
      Choice : Node_Id;
      Expr   : Node_Id;

      New_Aggr : Node_Id;
      New_Expr_List : List_Id := New_List;

      Low  : Node_Id;
      High : Node_Id;

      Nb_Choices   : Nat := 0;
      --  number of choices in a named array aggregate

      Nb_Positions : Nat := 0;
      --  number of elements in the new array aggregate

      Others_Present : Boolean;
      Others_Expr    : Node_Id;

   begin
      if Present (Component_Associations (N)) then
         Assoc := Last (Component_Associations (N));
         Others_Present :=
           (Nkind (First (Choices (Assoc))) = N_Others_Choice);

         if Others_Present then
            Others_Expr    :=  Expression (Assoc);
            Transfer_Itypes (Others_Expr, To => N);
         end if;

      else
         Others_Present := False;
      end if;

      --  Process component associations

      if Present (Component_Associations (N))
        and then No (Expressions (N))
      then
         --  Count the overall number of choices

         Assoc := First (Component_Associations (N));
         while Present (Assoc) loop
            Choice := First (Choices (Assoc));
            while Present (Choice) loop
               Nb_Choices := Nb_Choices + 1;
               Choice := Next (Choice);
            end loop;

            Assoc := Next (Assoc);
         end loop;

         if Others_Present then
            Nb_Choices := Nb_Choices - 1;
         end if;

         Convert_Named_Associations : declare
            Table : Case_Table_Type (1 .. Nb_Choices);
            --  Used to sort all the different choice values

            Nb_Static_Choices : Nat := 0;

         begin
            Assoc := First (Component_Associations (N));
            while Present (Assoc) loop

               Choice := First (Choices (Assoc));
               while Present (Choice) loop
                  if Nkind (Choice) = N_Others_Choice then
                     exit;

                  else
                     Get_Index_Bounds (Choice, Low, High);
                  end if;

                  Nb_Static_Choices := Nb_Static_Choices + 1;
                  Table (Nb_Static_Choices).Choice_Lo   := Low;
                  Table (Nb_Static_Choices).Choice_Hi   := High;
                  Table (Nb_Static_Choices).Choice_Node := Expression (Assoc);

                  Choice := Next (Choice);
               end loop;

               Assoc := Next (Assoc);
            end loop;

            if Nb_Static_Choices > 0 then
               Sort_Case_Table (Table);
            end if;

            --  If others present, take aggregate bounds from index constraint

            if Others_Present then
               Get_Index_Bounds (Index, Low, High);

            --  Otherwise we must have (Nb_Static_Choices > 0)

            else
               High := Table (Nb_Static_Choices).Choice_Hi;
               Low  := Table (1).Choice_Lo;

            end if;

            Nb_Positions :=
              1 + UI_To_Int (Expr_Value (High) - Expr_Value (Low));

            Convert_To_Positional : declare
               Pos_Expr : array (1 .. Nb_Positions) of Node_Id;
               --  Will contain the positional expressions of the aggregate

               UI_Low : constant Uint := Expr_Value (Low);
               --  The index value of the first aggregate positional element

               UI_Choice_Low  : Uint;
               UI_Choice_High : Uint;
               --  The low and high value of an aggregate choice

            begin
               --  Initially store the default expression for all positional
               --  aggregate elements. Later on this value will be overwritten
               --  if a choice gives an explicit value for it

               if Others_Present then
                  for J in 1 .. Nb_Positions loop
                     Pos_Expr (J) := New_Copy_Tree (Others_Expr);
                  end loop;
               end if;

               for J in 1 .. Nb_Static_Choices loop
                  UI_Choice_Low  := Expr_Value (Table (J).Choice_Lo);
                  UI_Choice_High := Expr_Value (Table (J).Choice_Hi);

                  for K in 1 + UI_To_Int (UI_Choice_Low - UI_Low) ..
                           1 + UI_To_Int (UI_Choice_High - UI_Low)
                  loop
                     if K > Nb_Positions then
                        exit;
                     else
                        Pos_Expr (K) := New_Copy_Tree (Table (J).Choice_Node);
                     end if;
                  end loop;
               end loop;

               for J in 1 .. Nb_Positions loop
                  if Present (Next_Index (Index)) then
                     Append_To (New_Expr_List,
                       Rewrite_Array_Aggregate
                         (Pos_Expr (J), Next_Index (Index)));
                  else
                     Append_To (New_Expr_List, Pos_Expr (J));
                  end if;
               end loop;

            end Convert_To_Positional;
         end Convert_Named_Associations;

      --  Consider positional aggregates, with possible others choice.

      elsif Present (Expressions (N)) then
         Expr := First (Expressions (N));

         while Present (Expr) loop
            if Present (Next_Index (Index)) then
               Append_To (New_Expr_List,
                 Rewrite_Array_Aggregate (Expr, Next_Index (Index)));
            else
               Append_To (New_Expr_List, New_Copy (Expr));
            end if;

            Nb_Positions := Nb_Positions + 1;
            Expr := Next (Expr);
         end loop;

         --  Append at the end of the new positional associations the
         --  missing positional values represented by Others.

         if Others_Present then

            if Present (Next_Index (Index)) then
               Others_Expr :=
                 Rewrite_Array_Aggregate (Others_Expr, Next_Index (Index));
            end if;

            Get_Index_Bounds (Index, Low, High);

            for J in 0 .. (UI_To_Int (Expr_Value (High) - Expr_Value (Low))
                           - Nb_Positions)
            loop
               Append (New_Copy_Tree (Others_Expr), New_Expr_List);
            end loop;
         end if;
      end if;

      New_Aggr := Make_Aggregate (Sloc (N), Expressions => New_Expr_List);
      Set_Etype (New_Aggr, Any_Composite);
      return New_Aggr;
   end Rewrite_Array_Aggregate;

   ----------------
   -- Build_Loop --
   ----------------

   function Build_Loop
     (N    : Node_Id;
      Ind  : Node_Id;
      L    : List_Id)
      return List_Id
   is
      Loc        : constant Source_Ptr := Sloc (N);
      Posit_List : constant List_Id := Expressions (N);
      Assoc_List : constant List_Id := Component_Associations (N);

      Assg_Stat_List : List_Id;
      New_Iden_List  : List_Id;
      R              : List_Id := New_List;

      Posit     : Node_Id;
      Assoc     : Node_Id;
      Choice    : Node_Id;
      Iden      : Node_Id;
      Iter_Sche : Node_Id;
      Disc_Def  : Node_Id;
      Loop_Stat : Node_Id;

      Next_Ind : Node_Id := Next_Index (Ind);
      Ind_Lo   : Node_Id;
      Ind_Hi   : Node_Id;
      R_Node   : Node_Id;

      Nb_Comp : Uint;

   begin

      Get_Index_Bounds (Ind, Ind_Lo, Ind_Hi);

      --  First deal with OTHERS in the aggregate

      if Present (Assoc_List) then
         Assoc := First (Assoc_List);

         while Present (Assoc) loop
            Choice := First (Choices (Assoc));

            while Present (Choice) loop
               if Nkind (Choice) = N_Others_Choice then

                  --  Create a new identifier list by copying the list of
                  --  identifiers visible at this point and adding a new
                  --  identifier corresponding to the defining identifier
                  --  just created. This list will be used as the indexed
                  --  component of the assignment concerning the temporary.

                  Iden :=
                    Make_Defining_Identifier (Loc,
                      Chars => New_Internal_Name ('I'));

                  R_Node :=
                    Make_Range (Loc,
                      Low_Bound  => New_Copy (Ind_Lo),
                      High_Bound => New_Copy (Ind_Hi));

                  Set_Etype (Iden, Etype (Ind_Lo));
                  New_Iden_List := New_List_Copy (L);
                  Append_To (New_Iden_List,
                    New_Reference_To (Iden, Loc));

                  --  Reset Analyzed flag for bounds because discriminants must
                  --  be expanded into discriminals

                  Set_Analyzed (High_Bound (R_Node), False);
                  Set_Analyzed (Low_Bound  (R_Node), False);

                  --  Create the code : for I__xxx in Ind_Lo .. Ind_Hi loop

                  Iter_Sche :=
                    Make_Iteration_Scheme (Loc,
                      Loop_Parameter_Specification =>
                        Make_Loop_Parameter_Specification (Loc,
                          Defining_Identifier => Iden,
                          Discrete_Subtype_Definition =>
                            Make_Subtype_Indication (Loc,
                              Subtype_Mark =>
                                New_Occurrence_Of (Etype (Ind_Lo), Loc),
                              Constraint =>
                                Make_Range_Constraint (Loc, R_Node))));

                  --  Deal with a multi-dimensional aggregate or create the
                  --  code : Tmp__xyz (New_Iden_List) := Expression (Assoc);

                  if Present (Next_Ind) then
                     Assg_Stat_List := Build_Loop (Expression (Assoc),
                                                   Next_Ind,
                                                   New_Iden_List);
                  else
                     Assg_Stat_List :=
                       New_List (
                         Make_Assignment_Statement (Loc,
                           Name =>
                             Make_Indexed_Component (Loc,
                               Prefix => New_Reference_To (Tmp, Loc),
                               Expressions => New_Iden_List),
                           Expression => Expression (Assoc)));
                  end if;


                  Loop_Stat :=
                    Make_Loop_Statement (Loc,
                      Identifier       => Empty,
                      Iteration_Scheme => Iter_Sche,
                      Statements       => Assg_Stat_List);
                  Append (Loop_Stat, R);
               end if;

               Choice := Next (Choice);
            end loop;

            Assoc := Next (Assoc);
         end loop;

         --  Named associations different from Others

         Assoc := First (Assoc_List);

         while Present (Assoc) loop
            Choice := First (Choices (Assoc));

            while Present (Choice) loop

               if Nkind (Choice) = N_Others_Choice then
                  null;

               --  Range choice

               elsif Nkind (Choice) = N_Range
                 or else (Is_Entity_Name (Choice)
                           and then Is_Type (Entity (Choice)))
               then

                  --  We need to build a loop that will instantiate
                  --  the temporary for the values covering the range
                  --  of the choice. So, we need a new defining identifier.
                  --  The code created here is similar to the code created
                  --  in the Others case.

                  if Nkind (Choice) = N_Range then
                     Disc_Def :=
                       Make_Subtype_Indication (Loc,
                         Subtype_Mark =>
                           New_Occurrence_Of (Etype (Ind_Lo), Loc),
                         Constraint =>
                           Make_Range_Constraint (Loc,
                             Make_Range (Loc,
                               Low_Bound  =>
                                 New_Copy (Low_Bound (Choice)),
                               High_Bound =>
                                 New_Copy (High_Bound (Choice)))));
                  else
                     Disc_Def := New_Reference_To (Entity (Choice), Loc);
                  end if;

                  Iden :=
                    Make_Defining_Identifier (Loc,
                      Chars => New_Internal_Name ('I'));
                  New_Iden_List := New_List_Copy (L);
                  Append_To (New_Iden_List, New_Reference_To (Iden, Loc));

                  Iter_Sche :=
                    Make_Iteration_Scheme (Loc,
                      Loop_Parameter_Specification =>
                        Make_Loop_Parameter_Specification (Loc,
                          Defining_Identifier => Iden,
                          Discrete_Subtype_Definition => Disc_Def));

                  if Present (Next_Ind) then
                     Assg_Stat_List :=
                       Build_Loop
                         (Expression (Assoc), Next_Ind, New_Iden_List);
                  else
                     Assg_Stat_List :=
                       New_List (
                         Make_Assignment_Statement (Loc,
                           Name =>
                             Make_Indexed_Component (Loc,
                               Prefix => New_Reference_To (Tmp, Loc),
                               Expressions => New_Iden_List),
                           Expression => Expression (Assoc)));
                  end if;

                  Loop_Stat :=
                    Make_Loop_Statement (Loc,
                      Identifier       => Empty,
                      Iteration_Scheme => Iter_Sche,
                      Statements       => Assg_Stat_List);

                  Append (Loop_Stat, R);

               --  Simple choice

               else
                  --  We just need to append the value of the choice to
                  --  the list of visible identifiers. In this case,
                  --  there is no loop created, just an assignment :
                  --  Tmp (Iden_List, Choice) := Expression (Assoc);

                  New_Iden_List := New_List_Copy (L);
                  Append (New_Copy (Choice), New_Iden_List);

                  if Present (Next_Ind) then
                     Assg_Stat_List := Build_Loop (Expression (Assoc),
                                                   Next_Ind,
                                                   New_Iden_List);
                  else
                     Assg_Stat_List :=
                       New_List (
                         Make_Assignment_Statement (Loc,
                           Name =>
                             Make_Indexed_Component (Loc,
                               Prefix => New_Reference_To (Tmp, Loc),
                               Expressions => New_Iden_List),
                           Expression => Expression (Assoc)));
                  end if;

                  Append_List (Assg_Stat_List, R);
               end if;

               Choice := Next (Choice);
            end loop;

            Assoc := Next (Assoc);
         end loop;
      end if;

      --  Positional associations

      --  We generate two different codes considering that the type of
      --  the index may be an enumeration type. if so, we will build
      --  a temporary whose value will be be updated by using each time
      --  the attribute 'succ of this temporary. Else, the value is
      --  directly passed in the indexed component list.

      if Present (Posit_List)
         and then Present (First (Posit_List))
      then
         Posit := First (Posit_List);
         Assg_Stat_List := New_List;
         New_Iden_List := New_List_Copy (L);
         Nb_Comp := Uint_0;

         --  If the type of the index is an enumeration type, create a new
         --  numbered identifier, add it to the the list of identifiers and
         --  create the object declaration code Iden : Index_Type := Ind_Lo;
         --  Else, append the low bound of the index to the list of
         --  indentifiers.

         New_Iden_List := New_List_Copy (L);
         if Is_Enumeration_Type (Etype (Ind_Lo)) then
            Iden :=
              Make_Defining_Identifier (Loc, New_Internal_Name ('X'));
            Append (New_Reference_To (Iden, Sloc (N)), New_Iden_List);
            Append_To (Assg_Stat_List,
              Make_Object_Declaration (Sloc (N),
                Defining_Identifier => Iden,
                Object_Definition =>
                  New_Reference_To (Etype (Ind_Lo), Sloc (N)),
                Expression => New_Copy (Ind_Lo)));

         else
            Append (New_Copy (Ind_Lo), New_Iden_List);
         end if;

         --  If the next index is present, then we deal with a multi
         --  dimensional aggregate and so call recursively Build_Loop.
         --  Else, we create the following code for the assignment:

         --    Tmp (New_Iden_List) := Posit;

         if Present (Next_Ind) then
            Append_List (Build_Loop (Posit, Next_Ind, New_Iden_List),
                         Assg_Stat_List);

         else
            Append_To (Assg_Stat_List,
              Make_Assignment_Statement (Sloc (N),
                Name =>
                  Make_Indexed_Component (Sloc (N),
                    Prefix => New_Reference_To (Tmp, Sloc (N)),
                    Expressions => New_Iden_List),
                Expression => New_Copy (Posit)));
         end if;

         Posit := Next (Posit);

         while Present (Posit) loop
            Nb_Comp := Nb_Comp + 1;

            --  In the index is an enumeration type then, create
            --  the assignment:

            --    Iden := Enum'Succ (Iden);

            --  Else, initialize a copy of the identifier list and
            --  add to this copy the code:

            --    Ind_Lo + Nb_Comp;

            if Is_Enumeration_Type (Etype (Ind_Lo)) then
               Append_To (Assg_Stat_List,
                 Make_Assignment_Statement (Loc,
                   Name       => New_Reference_To (Iden, Loc),
                   Expression =>
                     Make_Attribute_Reference (Loc,
                       Prefix => New_Reference_To (Etype (Ind_Lo), Loc),
                       Attribute_Name => Name_Succ,
                       Expressions => New_List (
                         New_Reference_To (Iden, Loc)))));

            else
               New_Iden_List := New_List_Copy (L);
               Append_To (New_Iden_List,
                 Make_Op_Add (Loc,
                   Left_Opnd  => New_Copy (Ind_Lo),
                   Right_Opnd => Make_Integer_Literal (Loc, Nb_Comp)));
            end if;

            if Present (Next_Ind) then
               Append_List (Build_Loop (Posit, Next_Ind, New_Iden_List),
                            Assg_Stat_List);

            else
               Append_To (Assg_Stat_List,
                 Make_Assignment_Statement (Loc,
                   Name =>
                     Make_Indexed_Component (Loc,
                       Prefix => New_Reference_To (Tmp, Loc),
                       Expressions => New_Iden_List),
                   Expression => New_Copy (Posit)));
            end if;

            Posit := Next (Posit);
         end loop;

         Append_List (Assg_Stat_List, R);
      end if;

      return R;
   end Build_Loop;

   ----------------
   -- Fill_Index --
   ----------------

   function Fill_Index (N : Node_Id; Ind : Node_Id) return List_Id is
      L : List_Id := New_List;

      Posit_List : constant List_Id := Expressions (N);
      Assoc_List : constant List_Id := Component_Associations (N);

      Posit  : Node_Id;
      Assoc  : Node_Id;
      Choice : Node_Id;
      Agg_Lo : Node_Id;
      Agg_Hi : Node_Id;

      Next_Ind : Node_Id := Next_Index (Ind);
      Ind_Lo   : Node_Id;
      Ind_Hi   : Node_Id;

      Agg_Lo_Val : Uint;
      Agg_Hi_Val : Uint;
      Nb_Comp    : Uint := Uint_0;

   begin
      --  Named associations

      if Present (Assoc_List) then

         --  Look for the first choice of the first association and assign
         --  the low bound and high bound of a range choice to respectively
         --  Agg_Lo ans Agg_Hi. if that's a simple choice then assign it to
         --  Agg_Lo and Agg_Hi.

         Assoc := First (Assoc_List);
         Choice := First (Choices (Assoc));

         if Nkind (Choice) = N_Range then
            Agg_Lo := Low_Bound (Choice);
            Agg_Hi := High_Bound (Choice);

         else
            Agg_Lo := Choice;
            Agg_Hi := Choice;
         end if;

         --  If the values are not static then there is only one association
         --  with only one choice. Nothing else is needed, we know the low
         --  high bound that we need for the index of the array. Else, we
         --  need to examine all the choices of all the associations to find
         --  the low and high bound. Note, that we are sure that there's no
         --  Others choice because the type of the aggregate is an
         --  unconstrained array.

         if Is_OK_Static_Expression (Agg_Lo)
           and then Is_OK_Static_Expression (Agg_Hi)
         then
            Agg_Lo_Val := Expr_Value (Agg_Lo);
            Agg_Hi_Val := Expr_Value (Agg_Hi);

            while Present (Assoc) loop
               Choice := First (Choices (Assoc));

               while Present (Choice) loop
                  if Nkind (Choice) = N_Range then
                     if Expr_Value (Low_Bound (Choice)) < Agg_Lo_Val then
                        Agg_Lo := Low_Bound (Choice);
                        Agg_Lo_Val := Expr_Value (Agg_Lo);
                     end if;

                     if Expr_Value (High_Bound (Choice)) > Agg_Hi_Val then
                        Agg_Hi := High_Bound (Choice);
                        Agg_Hi_Val := Expr_Value (Agg_Hi);
                     end if;

                  else
                     if Expr_Value (Choice) < Agg_Lo_Val then
                        Agg_Lo := Choice;
                        Agg_Lo_Val := Expr_Value (Agg_Lo);
                     end if;

                     if Expr_Value (Choice) > Agg_Hi_Val then
                        Agg_Hi := Choice;
                        Agg_Hi_Val := Expr_Value (Agg_Hi);
                     end if;
                  end if;

                  Choice := Next (Choice);
               end loop;

               Assoc := Next (Assoc);
            end loop;
         end if;

         --  Create the code Agg_Lo .. Agg_Hi and we deal with a multi
         --  dimensional aggregate, call recursively Fill_Index.

         Append_To (L,
           Make_Range (Sloc (N),
             Low_Bound  => New_Copy (Agg_Lo),
             High_Bound => New_Copy (Agg_Hi)));

         if Present (Next_Ind) then
            Append_List
              (Fill_Index (Expression (First (Assoc_List)), Next_Ind), L);
         end if;

      --  Positional associations

      elsif Present (Posit_List) then

         --  Get the bounds of the index coresponding to the aggregate
         --  Compute the number of positional arguments.

         --  Note that we look directly to the second positional association
         --  because we are sure that there's at least one and we want this
         --  first one to correspond to a number of positional associations
         --  of zero.

         Get_Index_Bounds (Ind, Ind_Lo, Ind_Hi);
         Posit := Next (First (Posit_List));

         while Present (Posit) loop
            Nb_Comp := Nb_Comp + 1;
            Posit := Next (Posit);
         end loop;

            if Is_Enumeration_Type (Etype (Ind_Lo)) then

               --  Create the following code for enumeration type:

               --    Ind_Lo .. Enum'Val (Enum'Pos (Ind_Lo) + Nb_Comp)

               Append_To (L,
                 Make_Range (Sloc (N),
                    Low_Bound  => New_Copy (Ind_Lo),
                    High_Bound =>
                      Make_Attribute_Reference (Sloc (N),
                        Prefix =>
                          New_Reference_To (Etype (Ind_Lo), Sloc (N)),
                        Attribute_Name => Name_Val,
                        Expressions => New_List (
                          Make_Op_Add (Sloc (N),
                            Left_Opnd  =>
                              Make_Attribute_Reference (Sloc (N),
                                Prefix =>
                                  New_Reference_To (Etype (Ind_Lo), Sloc (N)),
                                Attribute_Name => Name_Pos,
                                Expressions => New_List (
                                  New_Copy (Ind_Lo))),
                            Right_Opnd =>
                              Make_Integer_Literal (Sloc (N),
                                Intval => Nb_Comp))))));

            else
               --  Else create the following code:

               --    Ind_Lo .. Ind_Lo + Nb_Comp

               Append_To (L,
                 Make_Range (Sloc (N),
                   Low_Bound =>  New_Copy (Ind_Lo),
                   High_Bound =>
                     Make_Op_Add (Sloc (N),
                       Left_Opnd  => New_Copy (Ind_Lo),
                       Right_Opnd =>
                         Make_Integer_Literal (Sloc (N), Nb_Comp))));
            end if;

         if Present (Next_Ind) then
            Append_List (Fill_Index (First (Posit_List), Next_Ind), L);
         end if;
      end if;

      return L;
   end Fill_Index;

end Exp_Aggr;
