------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                 I N T E R F A C E S . C . S T R I N G S                  --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.9 $                              --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- The GNAT library is free software; you can redistribute it and/or modify --
-- it under terms of the GNU Library General Public License as published by --
-- the Free Software  Foundation; either version 2, or (at your option) any --
-- later version.  The GNAT library is distributed in the hope that it will --
-- be useful, but WITHOUT ANY WARRANTY;  without even  the implied warranty --
-- of MERCHANTABILITY  or  FITNESS FOR  A PARTICULAR PURPOSE.  See the  GNU --
-- Library  General  Public  License for  more  details.  You  should  have --
-- received  a copy of the GNU  Library  General Public License  along with --
-- the GNAT library;  see the file  COPYING.LIB.  If not, write to the Free --
-- Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.        --
--                                                                          --
------------------------------------------------------------------------------

with System; use System;
with System.Address_To_Access_Conversions;

package body Interfaces.C.Strings is

   package Char_Access is new Address_To_Access_Conversions (char);

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Peek (From : chars_ptr) return char;
   pragma Inline (Peek);
   --  Given a chars_ptr value, obtain referenced character

   procedure Poke (Value : char; Into : chars_ptr);
   pragma Inline (Poke);
   --  Given a chars_ptr, modify referenced Character value

   function "+" (Left : chars_ptr; Right : size_t) return chars_ptr;
   pragma Inline ("+");
   --  Address arithmetic on chars_ptr value

   function Position_Of_Nul (Into : char_array) return size_t;
   --  Returns position of the first Nul in Into or Into'Last + 1 if none

   function C_Malloc (Size : size_t) return chars_ptr;
   pragma Import (C, C_Malloc, "malloc");

   procedure C_Free (Address : chars_ptr);
   pragma Import (C, C_Free, "free");

   ---------
   -- "+" --
   ---------

   function "+" (Left : chars_ptr; Right : size_t) return chars_ptr is
   begin
      return Left + chars_ptr (Right);
   end "+";

   ----------
   -- Free --
   ----------

   procedure Free (Item : in out chars_ptr) is
   begin
      if Item = Null_Ptr then
         return;
      end if;

      C_Free (Item);
      Item := Null_Ptr;
   end Free;

   --------------------
   -- New_Char_Array --
   --------------------

   function New_Char_Array (Chars : in char_array) return chars_ptr is
      Index   : size_t;
      Pointer : chars_ptr;

   begin
      --  Get index of position before null. This can be -1, which is OK!

      Index := Position_Of_Nul (Into => Chars) - 1;

      --  Returned value is length of signficant part + 1 for the nul character

      Pointer := C_Malloc ((Index - Chars'First + 1) + 1);
      Update (Item   => Pointer,
              Offset => 0,
              Chars  => Chars,
              Check  => False);
      return Pointer;
   end New_Char_Array;

   ----------------
   -- New_String --
   ----------------

   function New_String (Str : in String) return chars_ptr is
   begin
      return New_Char_Array (To_C (Str));
   end New_String;

   ----------
   -- Peek --
   ----------

   function Peek (From : chars_ptr) return char is
      use Char_Access;
   begin
      return To_Pointer (Address (To_Address (From))).all;
   end Peek;

   ----------
   -- Poke --
   ----------

   procedure Poke (Value : char; Into : chars_ptr) is
      use Char_Access;
   begin
      To_Pointer (Address (To_Address (Into))).all := Value;
   end Poke;

   ---------------------
   -- Position_Of_Nul --
   ---------------------

   function Position_Of_Nul (Into : char_array) return size_t is
   begin
      for J in Into'Range loop
         if Into (J) = nul then
            return J;
         end if;
      end loop;

      return Into'Last + 1;
   end Position_Of_Nul;

   ------------
   -- Strlen --
   ------------

   function Strlen (Item : in chars_ptr) return size_t is
      Item_Index : size_t := 0;

   begin
      if Item = Null_Ptr then
         raise Dereference_Error;
      end if;

      loop
         if Peek (Item + Item_Index) = nul then
            return Item_Index;
         end if;

         Item_Index := Item_Index + 1;
      end loop;
   end Strlen;

   ------------------
   -- To_Chars_Ptr --
   ------------------

   function To_Chars_Ptr
     (Item      : in char_array_access;
      Nul_Check : in Boolean := False)
      return      chars_ptr
   is
   begin
      if Item = null then
         return Null_Ptr;
      elsif Nul_Check
        and then Position_Of_Nul (Into => Item.all) > Item'Last
      then
         raise Terminator_Error;
      else
         return To_Integer (Item (Item'First)'Address);
      end if;
   end To_Chars_Ptr;

   ------------
   -- Update --
   ------------

   procedure Update
     (Item   : in chars_ptr;
      Offset : in size_t;
      Chars  : in char_array;
      Check  : Boolean := True)
   is
      Index : chars_ptr := Item + Offset;

   begin
      if Check and then Offset + Chars'Length  > Strlen (Item) then
         raise Update_Error;
      end if;

      for J in Chars'Range loop
         Poke (Chars (J), Into => Index);
         Index := Index + 1;
      end loop;
   end Update;

   procedure Update
     (Item   : in chars_ptr;
      Offset : in size_t;
      Str    : in String;
      Check  : in Boolean := True)
   is
   begin
      Update (Item, Offset, To_C (Str), Check);
   end Update;

   -----------
   -- Value --
   -----------

   function Value (Item : in chars_ptr) return char_array is
      Result : char_array (0 .. Strlen (Item));

   begin
      if Item = Null_Ptr then
         raise Dereference_Error;
      end if;

      --  Note that the following loop will also copy the terminating Nul

      for J in Result'Range loop
         Result (J) := Peek (Item + J);
      end loop;

      return Result;
   end Value;

   function Value
     (Item   : in chars_ptr;
      Length : in size_t)
      return   char_array
   is
      Result : char_array (0 .. Length - 1);

   begin
      if Item = Null_Ptr then
         raise Dereference_Error;
      end if;

      for J in Result'Range loop
         Result (J) := Peek (Item + J);
         if Result (J) = nul then
            return Result (0 .. J);
         end if;
      end loop;

      return Result;
   end Value;

   function Value (Item : in chars_ptr) return String is
   begin
      return To_Ada (Value (Item));
   end Value;

   function Value (Item : in chars_ptr; Length : in size_t) return String is
   begin
      return To_Ada (Value (Item, Length));
   end Value;

end Interfaces.C.Strings;
