------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ E V A L                              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.33 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains various subprograms involved in compile time
--  evaluation of expressions and checks for staticness of expressions
--  and types. It also contains the circuitry for checking for violations
--  of pure and preelaborated conditions (this naturally goes here, since
--  these rules involve consideration of staticness).

--  Note: the static evaluation for attributes is found in Sem_Attr even
--  though logically it belongs here. We have done this so that it is easier
--  to add new attributes to GNAT.

with Types;  use Types;
with Uintp;  use Uintp;
with Urealp; use Urealp;

package Sem_Eval is

   ------------------------------------
   -- Handling of Static Expressions --
   ------------------------------------

   --  This package contains a set of routine that process individual
   --  subexpression nodes with the objective of folding (precomputing)
   --  the value of static expressions that are known at compile time and
   --  properly computing the setting of two flags that appear in every
   --  subexpression node:

   --    Is_Static_Expression

   --      This flag is set on any expression that is static according
   --      to the rules in (RM 4.9(3-14)).

   --    Raises_Constraint_Error

   --      This flag indicatest that it is known at compile time that the
   --      evaluation of an expression raises constraint error. If the
   --      expression is static, and this flag is off, then it is also known
   --      at compile time that the expression does not raise constraint error
   --      (i.e. the flag is accurate for static expressions, and conservative
   --      for non-static expressions.

   --  If a static expression does not raise constraint error, then the
   --  Raises_Constraint_Error flag is off, and the expression must be
   --  computed at compile time, which means that it has the form of either
   --  a literal, or a constant that is itself (recursively) either a literal
   --  or a constant.

   --  The above rules must be followed exactly in order for legality
   --  checks to be accurate. For subexpressions that are not static
   --  according to the RM definition, they are sometimes folded anyway,
   --  but of course in this case Is_Static_Expression is not set.

   -----------------
   -- Subprograms --
   -----------------

   procedure Check_Non_Static_Context (N : Node_Id);
   --  Deals with the special check required for a static expression that
   --  appears in a non-static context, i.e. is not part of a larger static
   --  expression (see RM 4.9(35)), i.e. the value of the expression must be
   --  within the base range of the base type of its expected type. A check
   --  is also made for expressions that are inside the base range, but
   --  outside the range of the expected subtype (this is a warning message
   --  rather than an illegality).
   --
   --  Note: most cases of non-static context checks are handled within
   --  Sem_Eval itself, including all cases of expressions at the outer
   --  level (i.e. those that are not a subexpression). Currently the
   --  only outside customers for this procedure are Sem_Attr (because
   --  Eval_Attribute is there), and Sem_Aggr (because aggregate processing
   --  intertwines analysis and expansion in a way that makes it hard to
   --  separate out a clean Eval_Aggregate routine).

   function Expr_Value (N : Node_Id) return Uint;
   --  Returns the folded value of the expression. This function is called
   --  in instances where it has already been determined that the expression
   --  is static. This version is used for integer values, and enumeration or
   --  character literals. In the latter two cases, the value returned is the
   --  Pos value in the relevant enumeration type. It can also be used for
   --  fixed-point values, in which case it returns the corresponding integer
   --  value. It cannot be used for floating-point values.

   function Expr_Value_E (N : Node_Id) return Entity_Id;
   --  Returns the folded value of the expression. This function is called
   --  in instances where it has already been determined that the expression
   --  is static. This version is used for enumeration types and returns the
   --  corresponding enumeration literal.

   function Expr_Value_R (N : Node_Id) return Ureal;
   --  Returns the folded value of the expression. This function is called
   --  in instances where it has already been determined that the expression
   --  is static. This version is used for real values (including both the
   --  floating-point and fixed-point cases). In the case of a fixed-point
   --  type, the real value is returned (cf above version returning Uint).

   function Expr_Value_S (N : Node_Id) return String_Id;
   --  Returns the folded value of the expression. This function is called
   --  in instances where it has already been determined that the expression
   --  is static. This version is used for string types and returns the
   --  corresponding string id value.

   procedure Eval_Actual                 (N : Node_Id);
   procedure Eval_Aggregate              (N : Node_Id);
   procedure Eval_Allocator              (N : Node_Id);
   procedure Eval_Arithmetic_Op          (N : Node_Id);
   procedure Eval_Character_Literal      (N : Node_Id);
   procedure Eval_Concatenation          (N : Node_Id);
   procedure Eval_Conditional_Expression (N : Node_Id);
   procedure Eval_Entity_Name            (N : Node_Id);
   procedure Eval_Indexed_Component      (N : Node_Id);
   procedure Eval_Integer_Literal        (N : Node_Id);
   procedure Eval_Logical_Op             (N : Node_Id);
   procedure Eval_Membership_Op          (N : Node_Id);
   procedure Eval_Named_Integer          (N : Node_Id);
   procedure Eval_Named_Real             (N : Node_Id);
   procedure Eval_Op_Expon               (N : Node_Id);
   procedure Eval_Op_Not                 (N : Node_Id);
   procedure Eval_Real_Literal           (N : Node_Id);
   procedure Eval_Relational_Op          (N : Node_Id);
   procedure Eval_Shift                  (N : Node_Id);
   procedure Eval_Short_Circuit          (N : Node_Id);
   procedure Eval_Slice                  (N : Node_Id);
   procedure Eval_String_Literal         (N : Node_Id);
   procedure Eval_Qualified_Expression   (N : Node_Id);
   procedure Eval_Type_Conversion        (N : Node_Id);
   procedure Eval_Unary_Op               (N : Node_Id);
   procedure Eval_Unchecked_Conversion   (N : Node_Id);

   procedure Fold_Str (N : Node_Id; Val : String_Id);
   --  Rewrite N with a new N_String_Literal node as the result of the
   --  compile time evaluation of the node N. Val is the resulting string
   --  value from the folding operation. The Is_Static_Expression flag is
   --  set in the result node.

   procedure Fold_Uint (N : Node_Id; Val : Uint);
   --  Rewrite N with a (N_Integer_Literal, N_Identifier, N_Character_Literal)
   --  node as the result of the compile time evaluation of the node N. Val
   --  is the result in the integer case and is the position of the literal
   --  in the literals list for the enumeration case. Is_Static_Expression
   --  is set True in the result node.

   procedure Fold_Ureal (N : Node_Id; Val : Ureal);
   --  Rewrite N with a new N_Real_Literal node as the result of the compile
   --  time evaluation of the node N. Val is the resulting real value from
   --  the folding operation. The Is_Static_Expression flag is set in the
   --  result node.

   function Is_OK_Static_Expression (N : Node_Id) return Boolean;
   --  An OK static expression is one that is static in the RM definition
   --  sense and which does not raise constraint error. This is used in
   --  contexts where the test is for compile time evaluation purposes.
   --  For legality checking purposes, always use Is_Static_Expression.

   function Is_Out_Of_Range (N : Node_Id; Typ : Entity_Id) return Boolean;
   --  See if it can be determined at compile time that the expression
   --  represented by node N is known to be out of range of the subtype
   --  Typ. True is returned if Typ is a scalar type, at least one of
   --  whose bounds is static, and N is a static expression which can
   --  be determined to be outside a static bound. Note that Typ does
   --  not have to be a static subtype, since there are cases where N
   --  is clearly outside the range of one bound, even though the other
   --  bound is non-static. A result of False does not mean that the
   --  expression is in range, merely that it cannot be determined at
   --  compile time that it is out of range.

   function Is_Static_Range (N : Node_Id) return Boolean;
   --  Determine if range is static, as defined in RM 4.9(26). The only
   --  allowed argument is an N_Range node (but note that the semantic
   --  analysis of equivalent range attribute references already turned
   --  them into the equivalent range).

   function Is_OK_Static_Range (N : Node_Id) return Boolean;
   --  Like Is_Static_Range, but also makes sure that the bounds of the
   --  range are compile-time evaluable (i.e. do not raise constraint error).
   --  A result of true means that the bounds are compile time evaluable.
   --  A result of false means they are not (either because the range is
   --  not static, or because one or the other bound raises CE).

   function Is_Static_Subtype (Typ : Entity_Id) return Boolean;
   --  Determines whether a subtype fits the definition of an Ada static
   --  subtype as given in (RM 4.9(26)).

   function Is_OK_Static_Subtype (Typ : Entity_Id) return Boolean;
   --  Like Is_Static_Subtype but also makes sure that the bounds of the
   --  subtype are compile-time evaluable (i.e. do not raise constraint
   --  error). A result of true means that the bounds are compile time
   --  evaluable. A result of false means they are not (either because the
   --  range is not static, or because one or the other bound raises CE).

   function Subtypes_Statically_Match (T1, T2 : Entity_Id) return Boolean;
   --  Determine whether two types T1, T2, which have the same base type,
   --  are statically matching subtypes (RM 4.9.1(1-2)).

private
   --  The Eval routines are all marked inline, since they are called once

   pragma Inline (Eval_Actual);
   pragma Inline (Eval_Allocator);
   pragma Inline (Eval_Character_Literal);
   pragma Inline (Eval_Conditional_Expression);
   pragma Inline (Eval_Indexed_Component);
   pragma Inline (Eval_Integer_Literal);
   pragma Inline (Eval_Named_Integer);
   pragma Inline (Eval_Named_Real);
   pragma Inline (Eval_Real_Literal);
   pragma Inline (Eval_Shift);
   pragma Inline (Eval_Slice);
   pragma Inline (Eval_String_Literal);
   pragma Inline (Eval_Unchecked_Conversion);

   pragma Inline (Is_OK_Static_Expression);

end Sem_Eval;
