/*
 *  Apple II emulator by Alexander Jean-Claude Bottema (C) 1994
 *
 *  $Id: misc.c,v 1.18 1998/02/12 04:19:51 chernabog Exp $
 *
 *  MODIFICATION HISTORY
 *   v0.3 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jan 1997.
 *   v0.4 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jun 1997.
 *   v0.5 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Feb 1998.
 *     This code has nothing to do with my employer, GTE Internetworking,
 *     BBN Technologies.  It was written completely on my own time and on
 *     my own machine.
 *
 **/

#include "defs.H"
#include "colors.h"

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <vga.h>
#include <unistd.h>
#include <sys/ioctl.h>
/* #include <sys/signal.h> */
#include <vgakeyboard.h>


/* different than in defs.H! */
#define C_Flag_6502	0x1
#define X_Flag_6502	0x20
#define I_Flag_6502	0x4
#define V_Flag_6502	0x40
#define B_Flag_6502	0x10
#define D_Flag_6502	0x8
#define Z_Flag_6502	0x2
#define N_Flag_6502	0x80

/* ----------------------------------
    External functions
   ---------------------------------- */

/* in apple2.S */
extern unsigned char read_keyboard();

extern void	cpu65x02(),
		assign_6502_table(),
#ifdef APPLE_IIE
		assign_65c02_table(),
#endif
#ifdef II_UNDOCUMENTED
		assign_6502_undoc_table(),
#endif
		read_ram_default(),
		write_ram_default(),
		write_ram_text_page0(),
		write_ram_text_mixed0(),
		write_ram_text_page1(),
		write_ram_text_mixed1(),
		write_ram_hires_page0_even(),
		write_ram_hires_page0_odd(),
		write_ram_hires_mixed0_even(),
		write_ram_hires_mixed0_odd(),
		write_ram_hires_page1_even(),
		write_ram_hires_page1_odd(),
		write_ram_hires_mixed1_even(),
		write_ram_hires_mixed1_odd(),

		ram_nop(),
		read_keyboard_strobe(),
		read_random(),
		read_speaker_toggle_pc(),
		read_switch_primary_page(),
		read_switch_secondary_page(),
		read_switch_graphics(),
		read_switch_text(),
		read_switch_no_mixed(),
		read_switch_mixed(),
		read_switch_lores(),
		read_switch_hires(),
		update_video_screen(),

		read_button0(),
		read_button1(),
		read_button2(),
		read_gc0(),
		read_gc1(),
		read_gc_strobe(),
    
		lc_c080(),
		lc_c081(),
		lc_c082(),
		lc_c083(),
		lc_c088(),
		lc_c089(),
		lc_c08a(),
		lc_c08b(),
		write_ram_bank(),
		read_ram_bank(),
		write_ram_lc(),
		read_ram_lc();

#ifdef APPLE_IIE
extern void	iie_write_ram_default(),
		iie_read_ram_default(),

		/* //e text pages */
		iie_read_ram_text_page0(),
		iie_write_ram_text_page0(),
		iie_write_ram_text_mixed0(),
		iie_read_screen_hole_text_page0(),
		iie_write_screen_hole_text_page0(),
		iie_write_ram_text_page1(),
		iie_write_ram_text_mixed1(),

		/* //e hires page 0 */
		iie_read_ram_hires_page0(),
		iie_write_ram_hires_page0_odd(),
		iie_write_ram_hires_page0_even(),
		iie_write_ram_hires_mixed0_odd(),
		iie_write_ram_hires_mixed0_even(),
		iie_read_screen_hole_hires_page0(),
		iie_write_screen_hole_hires_page0(),

		/* //e hires page 1 */
		iie_write_ram_hires_page1_odd(),
		iie_write_ram_hires_page1_even(),
		iie_write_ram_hires_mixed1_odd(),
		iie_write_ram_hires_mixed1_even(),

		/* //e zpage,stack, ram banks */
		iie_read_ram_zpage_and_stack(),
		iie_write_ram_zpage_and_stack(),
		iie_read_ram_lc(),
		iie_write_ram_lc(),
		iie_read_ram_bank(),
		iie_write_ram_bank(),
		iie_read_slot3(),
		iie_read_slot6(),
		iie_read_slotx(),
		iie_read_slot_expansion(),
		iie_disable_slot_expansion(),
		iie_read_gc2(),
		iie_read_gc3(),

		/* //e toggle softswitches */
		iie_ramrd_main(),
		iie_ramrd_aux(),
		iie_ramwrt_main(),
		iie_ramwrt_aux(),
		iie_80store_off(),
		iie_80store_on(),
		iie_altzp_main(),
		iie_altzp_aux(),
		iie_80col_off(),
		iie_80col_on(),
		iie_altchar_off(),
		iie_altchar_on(),
		iie_c3rom_peripheral(),
		iie_c3rom_internal(),
		iie_cxrom_peripheral(),
		iie_cxrom_internal(),
		iie_ioudis_on(),
		iie_ioudis_off(),
		iie_dhires_on(),
		iie_dhires_off(),
		iie_hires_off(),
		iie_hires_on(),
		iie_page2_on(),
		iie_page2_off(),

		/* //e check softswitche settings */
		iie_check_80store(),
		iie_check_bank(),
		iie_check_lcram(),
		iie_check_ramrd(),
		iie_check_ramwrt(),
		iie_check_altzp(),
		iie_check_c3rom(),
		iie_check_cxrom(),
		iie_check_80col(),
		iie_check_altchar(),
		iie_check_text(),
		iie_check_mixed(),
		iie_check_hires(),
		iie_check_page2(),
		iie_check_ioudis(),
		iie_check_dhires(),
		iie_check_vbl();
#endif

/* in diskio.S */
extern void	disk_read_nop(),
		disk_read_phase(),
		disk_read_motor_off(),
		disk_read_motor_on(),
		disk_read_select_a(),
		disk_read_select_b(),
		disk_read_byte(),
		disk_read_latch(),
		disk_write_latch(),
		disk_read_prepare_in(),
		disk_read_prepare_out();

/* in disk.c */
extern void c_init_6();

/* in interface.c */
void c_load_interface_font();

/* in keys.c */
extern void c_lock_menu_mode_on();
extern void c_lock_menu_mode_off();
extern void c_wait_menu_buf_empty();
extern void c_initialize_keyboard();
extern void c_initialize_mouse();
extern void c_read_keyboard_strobe();

/* in joystick.c */
#ifdef PC_JOYSTICK
extern void c_calculate_joystick_parms();
extern void c_open_joystick();
extern short js_center_x, js_center_y, js_max_x, js_max_y, js_min_x, js_min_y;
extern long js_timelimit;
#endif

/* ----------------------------------
    graphics globals
   ---------------------------------- */
unsigned short	  video_line_offset[24] =
		   { 0x000, 0x080, 0x100, 0x180, 0x200, 0x280, 0x300, 0x380,
		     0x028, 0x0A8, 0x128, 0x1A8, 0x228, 0x2A8, 0x328, 0x3A8,
		     0x050, 0x0D0, 0x150, 0x1D0, 0x250, 0x2D0, 0x350, 0x3D0 };

int		  chipset;			/* VGA chipset		   */
int		  page_2_offset;		/* SVGA page 2 screen off. */
unsigned char     *GM;	     /* --- Base address of graphic area --- */
unsigned char     *svga_GM;  /* --- SVGA base address of graphic area --- */

Tr		  normal_vga_mode = False;
Tr		  force_vga_mode = False;

unsigned char     char_rom[0x800];		/* --- Character ROM --- */
unsigned char     expanded_font[0x800*8];	/* --- Font --- */

/* --- Precalculated color bytes for hires. --- */
unsigned char     expanded_col_hires_even[0x100*8]; 
unsigned char     expanded_col_hires_odd[0x100*8]; 
unsigned char odd_colors[2] = {COLOR_LIGHT_PURPLE, COLOR_LIGHT_BLUE};
unsigned char even_colors[2] = {COLOR_LIGHT_GREEN, COLOR_LIGHT_RED};

/* --- Precalculated text page rows given offset addr --- */
unsigned char     text_page_rows[1 K];
unsigned char	  text_page_cols[1 K];

/* --- Precalculated hi-res page SVGA offsets given addr --- */
unsigned short hires_page_offset[8 K];
unsigned char hires_col_offset[8 K];

#ifdef APPLE_IIE
/* --- Precalculated double hires auxiliary page SVGA offsets given addr --- */
unsigned short	  dhires_aux_page_offset[8 K];

unsigned char dhires_colors1[256] = {
    0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0x8,0x9,0xa,0xb,0xc,0xd,0xe,0xf,
    0x0,0x1,0x3,0x3,0x5,0x5,0x7,0x7,0x9,0x9,0xb,0xb,0xd,0xd,0xf,0xf,
    0x0,0x1,0x2,0x3,0x6,0x5,0x6,0x7,0xa,0x9,0xa,0xb,0xe,0xd,0xe,0xf,
    0x0,0x1,0x3,0x3,0x7,0x5,0x7,0x7,0xb,0x9,0xb,0xb,0xf,0xd,0xf,0xf,
    0x0,0x1,0x2,0x3,0x4,0x4,0x6,0x7,0x8,0x9,0xa,0xb,0xc,0xd,0xe,0xf,
    0x0,0x1,0x3,0x3,0x5,0x5,0x7,0x7,0xd,0x9,0xb,0xb,0xd,0xd,0xf,0xf,
    0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0xe,0x9,0xa,0xb,0xe,0xd,0xe,0xf,
    0x0,0x1,0x7,0x3,0x7,0x5,0x7,0x7,0xf,0x9,0xb,0xb,0xf,0xd,0xf,0xf,
};

unsigned char dhires_colors2[256] = {
    0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x8,0x8,0x0,0xb,0x8,0xd,0x0,0x0,
    0x1,0x1,0x1,0x1,0x0,0x5,0x1,0x1,0x0,0x9,0xb,0xb,0x0,0xd,0xf,0xf,
    0x0,0x1,0x2,0x2,0x2,0x5,0x2,0x2,0x0,0xa,0xa,0xa,0xe,0xd,0x2,0x2,
    0x3,0x3,0x3,0x3,0x7,0x5,0x7,0x7,0x0,0xb,0xb,0xb,0xf,0xd,0xf,0xf,
    0x0,0x0,0x4,0x0,0x4,0x4,0x4,0x4,0xc,0x8,0x4,0x8,0xc,0xd,0x4,0x4,
    0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0xd,0x4,0x4,0x4,0xd,0xd,0x4,0x4,
    0x6,0x6,0x6,0x2,0xe,0x6,0x6,0x6,0xe,0xe,0xa,0xa,0xe,0x6,0xe,0x6,
    0x7,0x7,0x7,0x7,0x7,0x7,0x7,0x7,0xf,0xf,0xb,0xb,0xf,0xf,0xf,0xf,
};
#endif

unsigned char	  vmode_text;
unsigned char	  vmode_mixed;
unsigned char	  vmode_page2;
unsigned char	  vmode_hires;
unsigned char	  vmode_active;

unsigned char	vga_mem_page_0[ 64 K ];  /* Only used in standard VGA */
unsigned char	vga_mem_page_1[ 64 K ];	 /* Only used in standard VGA */

int		crnt_visual_svga_page = -1; /* Current visual SVGA page */
int		crnt_active_svga_page = -2; /* Current active SVGA page */
int		oldpage = -1;
int		oldscreen = -1;

/* ----------------------------------
    global apple2 variables
   ---------------------------------- */
unsigned char	  table_encode_flags[256];
unsigned char	  table_decode_flags[256];


/* Apple //e variables */
#ifdef APPLE_IIE
int lc_offset = 0;
int zp_offset = 0;
int ramrd_offset = 0;
int ramwrt_offset = 0;
int write_text_page_offset = 0;
int read_text_page_offset = 0;
int write_hires_page_offset = 0;
int read_hires_page_offset = 0;
/* int page2_setting = 0; */
int c3rom_offset = 0;
int cxrom_offset = 0;
int c8rom_offset = 0x10000;
unsigned char slot_allocated;

unsigned char vmode_screen;			/* current screen */
unsigned char vmode_dhires;
unsigned char dhires_flag;
unsigned char eighty_store_flag;
unsigned char ramrd_flag;
unsigned char ramwrt_flag;
unsigned char altzp_flag;
unsigned char eighty_col_flag;
unsigned char altchar_flag;
unsigned char ioudis_flag;
unsigned char c3rom_flag;
unsigned char c8rom_flag;
unsigned char cxrom_flag;
#endif

int apple_mode = 0;	/* regular ][+ mode */
int table_opcodes[256];	/* current opcode table */

unsigned char	  apple_ii_64k[2][64 K];	/* 128 K */
Function	  table_read_memory[64 K];	/* 64 K */
Function	  table_write_memory[64 K];	/* 64 K */
unsigned char	  apple_ii_rom[12 K];
unsigned char	  apple_iie_rom[32 K];		/* //e */
unsigned char	  slot6_rom[256];
unsigned char	  language_card[2][8 K];
unsigned char	  language_banks[2][8 K];

unsigned char	  language_current_bank;
unsigned char	  language_card_second;
unsigned char	  language_card_write;
unsigned char	  language_card_read;

unsigned char	  exception_flag;
unsigned char	  current_slot;

unsigned short	  apple_speed = 1;
unsigned char	  color_mode = 0;
unsigned char	  strict_color = 0;
unsigned char	  sound_mode = 1;
unsigned char	  random_value = 0;

unsigned char	  system_path[ 2048 ] = ".";
unsigned char	  temp[ 4096 ];
uid_t user, privileged;

/* in keys.c */
extern short   	  joy_mode;
extern short	  joy_step;
extern short	  joy_center_x;
extern short	  joy_center_y;
extern short	  joy_range;

/* in interface.c */
extern unsigned char	disk_path[ 1024 ];

/* in prefs.l */
extern void lowercase_string(char *s);
extern FILE *prefsin;			/* yyin */
extern int prefslex();			/* yylex() */
extern int line_num;

#ifdef DEBUGGER
/* in debugger.c */
extern int breakpoints[];
extern int watchpoints[];
extern int op_breakpoints[256];
extern unsigned char watch_data;	/* new byte */
extern int watch_addrs;			/* address of tripped watchpoint */
#endif


/* -------------------------------------------------------------------------
    c_setpage(p):    Set SVGA 64k page and update GM
                     (Graph Memory base address)
   ------------------------------------------------------------------------- */
void c_setpage(int p)
{
/*     if (p == oldpage) */
/* 	return; */
    oldpage = p;

    if (!normal_vga_mode)
    {
        vga_setpage( p );
        svga_GM = GM = vga_getgraphmem();
    }
    else
    {
        c_lock_menu_mode_on();
        svga_GM = vga_getgraphmem();
	GM = (crnt_visual_svga_page == p) ? svga_GM :
	     ((p == 0) ? vga_mem_page_0 : vga_mem_page_1);
        c_lock_menu_mode_off();

        crnt_active_svga_page = p;
    }
}

/* -------------------------------------------------------------------------
    c_setscreen(p):    Switch to screen page p
   ------------------------------------------------------------------------- */
void c_setscreen(int m)
{
/*     if (m == oldscreen) */
/* 	return; */
    oldscreen = m;

    if (!normal_vga_mode)
        vga_setdisplaystart( m * page_2_offset );
    else
    {
        c_wait_menu_buf_empty();
        c_lock_menu_mode_on();
        if (m != crnt_visual_svga_page) {
	    memcpy( ((crnt_visual_svga_page == 0) ?
		        vga_mem_page_0 : vga_mem_page_1),
		    svga_GM, 64 K );
	    memcpy( svga_GM, ((m == 0) ? vga_mem_page_0 : vga_mem_page_1),
		    64 K );
/* 	    crnt_visual_svga_page = m; */
	    GM = (crnt_active_svga_page == m) ? svga_GM :
	             ((crnt_active_svga_page == 0) ?
		          vga_mem_page_0 : vga_mem_page_1);
	    crnt_visual_svga_page = m;
	}
        c_lock_menu_mode_off();
    }
}

/* -------------------------------------------------------------------------
    c_update_video_screen()
   ------------------------------------------------------------------------- */

void c_update_video_screen()
{
    c_wait_menu_buf_empty();
    vga_clear();

    if (normal_vga_mode) {
	memset( vga_mem_page_0, 0, 64 K );
	memset( vga_mem_page_1, 0, 64 K );
    }

    update_video_screen();
}

#ifdef APPLE_IIE
/* -------------------------------------------------------------------------
    c_initialize_dhires_values()
   ------------------------------------------------------------------------- */
void c_initialize_dhires_values() {
    int i;
/*     int value, v; */
/*     unsigned char locolor, hicolor; */

    /* precalculate the colors for all the 256*8 bit combinations. */
/*     for (value = 0x00, v = 0; value <= 0xFF; value++) { */
/* 	locolor = (value & 0x0F) | 0x10; */
/* 	hicolor = (value << 4) | 0x10; */

/* 	dhires_colors[v++] = locolor; */
/* 	dhires_colors[v++] = locolor; */
/* 	dhires_colors[v++] = locolor; */
/* 	dhires_colors[v++] = locolor; */

/* 	dhires_colors[v++] = hicolor; */
/* 	dhires_colors[v++] = hicolor; */
/* 	dhires_colors[v++] = hicolor; */
/* 	dhires_colors[v++] = hicolor; */
/*     } */

    for (i = 0; i < 0x80; i++) {
	dhires_colors1[i+0x80] = dhires_colors1[i];
	dhires_colors2[i+0x80] = dhires_colors2[i];
    }
}
#endif

/* -------------------------------------------------------------------------
    c_initialize_hires_values()
   ------------------------------------------------------------------------- */

void c_initialize_hires_values()
{
    int     value, b, v, e, /*color_toggle,*/ last_not_black;

    /* precalculate the colors for all the 256*8 bit combinations. */
    for (value = 0x00; value <= 0xFF; value++) {
	for (e = value * 8, last_not_black = False, v = value, b = 0;
	     b < 7; b++, v >>= 1, e++)
	{
	    if (v & 1)
	    {
		expanded_col_hires_even[ e ] = last_not_black ?
					    COLOR_LIGHT_WHITE :
					((b & 1) ?
					    ((value & 0x80) ?
					         COLOR_LIGHT_RED :
						 COLOR_LIGHT_GREEN) :
					    ((value & 0x80) ?
						 COLOR_LIGHT_BLUE :
						 COLOR_LIGHT_PURPLE));

		expanded_col_hires_odd[ e ] = last_not_black ?
					    COLOR_LIGHT_WHITE :
					((b & 1) ?
					    ((value & 0x80) ?
					         COLOR_LIGHT_BLUE :
						 COLOR_LIGHT_PURPLE) :
					    ((value & 0x80) ?
						 COLOR_LIGHT_RED :
						 COLOR_LIGHT_GREEN));

		if (last_not_black && b > 0)
		    expanded_col_hires_even[ e - 1 ] = COLOR_LIGHT_WHITE,
		    expanded_col_hires_odd[ e - 1 ] = COLOR_LIGHT_WHITE;

		last_not_black = True;
	    }
	    else
		expanded_col_hires_even[ e ] = COLOR_BLACK,
		expanded_col_hires_odd[ e ] = COLOR_BLACK,
	        last_not_black = False;
	}
    }
    if (color_mode == 0) {	/* Black and White */
        for (value = 0x00; value <= 0xFF; value++) {
	    for (b = 0, e = value * 8; b < 7; b++, e++)	{
		if (expanded_col_hires_even[ e ] != COLOR_BLACK)
		    expanded_col_hires_even[ e ] = COLOR_LIGHT_WHITE;
		if (expanded_col_hires_odd[ e ] != COLOR_BLACK)
		    expanded_col_hires_odd[ e ] = COLOR_LIGHT_WHITE;
	    }
	}
    }
    else if (color_mode == LAZY_INTERP) /* Lazy Interpolated color */
    {
        for (value = 0x00; value <= 0xFF; value++)
        {
            for (b = 1, e = value * 8 + 1; b <= 5; b += 2, e += 2)
            {
                if (expanded_col_hires_even[ e ] == COLOR_BLACK &&
                    expanded_col_hires_even[ e - 1 ] != COLOR_BLACK &&
                    expanded_col_hires_even[ e + 1 ] != COLOR_BLACK)
                      expanded_col_hires_even[ e ] =
                      expanded_col_hires_even[ e - 1 ];

                if (expanded_col_hires_odd[ e ] == COLOR_BLACK &&
                    expanded_col_hires_odd[ e - 1 ] != COLOR_BLACK &&
                    expanded_col_hires_odd[ e + 1 ] != COLOR_BLACK)
                      expanded_col_hires_odd[ e ] =
                      expanded_col_hires_odd[ e - 1 ];
            }

            for (b = 0, e = value * 8; b <= 6; b += 2, e += 2) {
                if (expanded_col_hires_odd[ e ] == COLOR_BLACK) {
                    if (b > 0 && b < 6) {
                    if (expanded_col_hires_even[e+1] != COLOR_BLACK &&
                        expanded_col_hires_even[e-1] != COLOR_BLACK &&
                        expanded_col_hires_even[e+1] != COLOR_LIGHT_WHITE &&
                        expanded_col_hires_even[e-1] != COLOR_LIGHT_WHITE)
                        expanded_col_hires_even[e] =
                            expanded_col_hires_even[e-1];
                    } else if (b == 0) {
                    if (expanded_col_hires_even[e+1] != COLOR_BLACK &&
                        expanded_col_hires_even[e+1] != COLOR_LIGHT_WHITE)
                        expanded_col_hires_even[e] =
                            expanded_col_hires_even[e+1];
                    } else {
                    if (expanded_col_hires_even[e-1] != COLOR_BLACK &&
                        expanded_col_hires_even[e-1] != COLOR_LIGHT_WHITE)
                        expanded_col_hires_even[ e ] =
                            expanded_col_hires_even[ e - 1 ];
                    }
                }

                if (expanded_col_hires_odd[ e ] == COLOR_BLACK) {
                    if (b > 0 && b < 6) {
                    if (expanded_col_hires_odd[e+1] != COLOR_BLACK &&
                        expanded_col_hires_odd[e-1] != COLOR_BLACK &&
                        expanded_col_hires_odd[e+1] != COLOR_LIGHT_WHITE &&
                        expanded_col_hires_odd[e-1] != COLOR_LIGHT_WHITE)
                        expanded_col_hires_odd[e] =
                            expanded_col_hires_odd[e-1];
                    }
                    else if (b == 0) {
                    if (expanded_col_hires_odd[e+1] != COLOR_BLACK &&
                        expanded_col_hires_odd[e+1] != COLOR_LIGHT_WHITE)
                        expanded_col_hires_odd[e] =
                            expanded_col_hires_odd[e+1];
                    }
                    else if
                        (expanded_col_hires_odd[e-1] != COLOR_BLACK &&
                         expanded_col_hires_odd[e-1] != COLOR_LIGHT_WHITE)
                        expanded_col_hires_odd[e] =
                            expanded_col_hires_odd[e-1];
                }
            }
        }
    }
    else if (color_mode == INTERP) {	/* Color and strict interpolation */
        for (value = 0x00; value <= 0xFF; value++) {
	    for (b = 1, e = value * 8 + 1; b <= 5; b += 2, e += 2) {
		if (expanded_col_hires_even[e] == COLOR_BLACK) {
		    if (expanded_col_hires_even[e-1] != COLOR_BLACK &&
			expanded_col_hires_even[e+1] != COLOR_BLACK &&
			expanded_col_hires_even[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] != COLOR_LIGHT_WHITE)
		      expanded_col_hires_even[e] =
			  expanded_col_hires_even[e-1];

		    else if (
			expanded_col_hires_even[e-1] != COLOR_BLACK &&
			expanded_col_hires_even[e+1] != COLOR_BLACK &&
			expanded_col_hires_even[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] == COLOR_LIGHT_WHITE)
		      expanded_col_hires_even[e] =
			  expanded_col_hires_even[e-1];

		    else if (
			expanded_col_hires_even[e-1] != COLOR_BLACK &&
			expanded_col_hires_even[e+1] != COLOR_BLACK &&
			expanded_col_hires_even[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] != COLOR_LIGHT_WHITE)
		      expanded_col_hires_even[e] =
			  expanded_col_hires_even[e+1];

		    else if (
			expanded_col_hires_even[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] == COLOR_LIGHT_WHITE)
		      expanded_col_hires_even[e] = (value & 0x80)
			  ? COLOR_LIGHT_BLUE : COLOR_LIGHT_PURPLE;

		}
		if (expanded_col_hires_odd[e] == COLOR_BLACK) {
		    if (expanded_col_hires_odd[e-1] != COLOR_BLACK &&
			expanded_col_hires_odd[e+1] != COLOR_BLACK &&
			expanded_col_hires_odd[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] != COLOR_LIGHT_WHITE)
		      expanded_col_hires_odd[e] =
			  expanded_col_hires_odd[e-1];

		    else if (
		        expanded_col_hires_odd[e-1] != COLOR_BLACK &&
			expanded_col_hires_odd[e+1] != COLOR_BLACK &&
			expanded_col_hires_odd[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] == COLOR_LIGHT_WHITE)
		      expanded_col_hires_odd[e] =
			  expanded_col_hires_odd[e-1];

		    else if (
		        expanded_col_hires_odd[e-1] != COLOR_BLACK &&
			expanded_col_hires_odd[e+1] != COLOR_BLACK &&
			expanded_col_hires_odd[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] != COLOR_LIGHT_WHITE)
		      expanded_col_hires_odd[e] =
			  expanded_col_hires_odd[e+1];

		    else if (
			expanded_col_hires_odd[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] == COLOR_LIGHT_WHITE)
			expanded_col_hires_odd[e] = (value & 0x80)
			    ? COLOR_LIGHT_RED : COLOR_LIGHT_GREEN;
		}
	    }

	    for (b = 0, e = value * 8; b <= 6; b += 2, e += 2) {
		if (expanded_col_hires_even[ e ] == COLOR_BLACK) {
		    if (b > 0 && b < 6) {
		    if (expanded_col_hires_even[e-1] != COLOR_BLACK &&
			expanded_col_hires_even[e+1] != COLOR_BLACK &&
			expanded_col_hires_even[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] != COLOR_LIGHT_WHITE)
			expanded_col_hires_even[e] =
			    expanded_col_hires_even[e-1];

		    else if (
			expanded_col_hires_even[e-1] != COLOR_BLACK &&
			expanded_col_hires_even[e+1] != COLOR_BLACK &&
			expanded_col_hires_even[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] == COLOR_LIGHT_WHITE)
			expanded_col_hires_even[e] =
			    expanded_col_hires_even[e-1];

		    else if (
			expanded_col_hires_even[e-1] != COLOR_BLACK &&
			expanded_col_hires_even[e+1] != COLOR_BLACK &&
			expanded_col_hires_even[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] != COLOR_LIGHT_WHITE)
			expanded_col_hires_even[e] =
			    expanded_col_hires_even[e+1];

		    else if (
			expanded_col_hires_even[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] == COLOR_LIGHT_WHITE)
			expanded_col_hires_even[e] = (value & 0x80)
			    ? COLOR_LIGHT_RED : COLOR_LIGHT_GREEN;
		    }
		}

		if (expanded_col_hires_odd[e] == COLOR_BLACK) {
		    if (b > 0 && b < 6) {
		    if (expanded_col_hires_odd[e-1] != COLOR_BLACK &&
			expanded_col_hires_odd[e+1] != COLOR_BLACK &&
			expanded_col_hires_odd[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] != COLOR_LIGHT_WHITE)
			expanded_col_hires_odd[e] =
			    expanded_col_hires_odd[e-1];

		    else if (
			expanded_col_hires_odd[e-1] != COLOR_BLACK &&
			expanded_col_hires_odd[e+1] != COLOR_BLACK &&
			expanded_col_hires_odd[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] == COLOR_LIGHT_WHITE)
			expanded_col_hires_odd[e] =
			    expanded_col_hires_odd[e-1];

		    else if (
			expanded_col_hires_odd[e-1] != COLOR_BLACK &&
			expanded_col_hires_odd[e+1] != COLOR_BLACK &&
			expanded_col_hires_odd[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] != COLOR_LIGHT_WHITE)
			expanded_col_hires_odd[e] =
			    expanded_col_hires_odd[e+1];

		    else if (
			expanded_col_hires_odd[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] == COLOR_LIGHT_WHITE)
			expanded_col_hires_odd[e] = (value & 0x80)
			    ? COLOR_LIGHT_BLUE : COLOR_LIGHT_PURPLE;
		    }
		}
	    }
	}
    }
}

/* -------------------------------------------------------------------------
    c_read_font_file() - loads the font file
   ------------------------------------------------------------------------- */
void c_read_font_file() {

    FILE *f;

    sprintf(temp, "%s/character.rom", system_path);
    if ((f = fopen(temp, "r")) == NULL)
    {
	printf("Cannot find file '%s'.\n", temp);
	exit(1);
    }
    fread( char_rom, 1, 0x800, f );
    fclose( f );
}


#ifdef APPLE_IIE
/* -------------------------------------------------------------------------
    c_set_iie_cursor(): set the cursor in //e mode
   ------------------------------------------------------------------------- */
void c_set_iie_cursor() {
    unsigned char *iie_cursor[] = {
	"       ",
	" x x x ",
	"  x x  ",
	" x x x ",
	"  x x  ",
	" x x x ",
	"       ",
	"       "
    };

    int i, j;
    int p = 0xFF * 8 * 8;

    for (i = 0; i < 8; i++, p++) {
	for (j = 0; j < 7; j++, p++) {
	    expanded_font[p] = (iie_cursor[i][j] == 'x') ?
		COLOR_LIGHT_WHITE : 0;
	}
    }
}

/* -------------------------------------------------------------------------
    c_set_altchar() - set alternate character set
   ------------------------------------------------------------------------- */
void c_set_altchar()
{
    int i, c, p, q;

    /* HACK - NEED TO PUT MOUSETEXT HERE! */
    for (c = 0x40, p = c * 64, q = (c + 0x40) * 64; c < 0x60; c++)
        for (i = c * 8; i < c * 8 + 64; i++, p++, q++)
	    expanded_font[ p ] = expanded_font[ q ];

    /* inverse lowercase letters */
    for (c = 0x60, p = c * 64, q = (c + 0x80) * 64; c < 0x80; c++)
        for (i = c * 8; i < c * 8 + 64; i++, p++, q++)
	    expanded_font[ p ] = (expanded_font[ q ] == COLOR_LIGHT_WHITE) ?
		0 : COLOR_LIGHT_WHITE;
}
#endif

/* -------------------------------------------------------------------------
    c_set_primary_char() - set primary character set
   ------------------------------------------------------------------------- */
void c_set_primary_char() {
    int i, c, p, q;

    /* flashing uppercase and special characters */
    for (c = 0x40, p = c * 64, q = (c + 0x40) * 64; c < 0x80; c++)
        for (i = c * 8; i < c * 8 + 64; i++, p++, q++)
	    expanded_font[ p ] = (expanded_font[ q ] == COLOR_LIGHT_WHITE) ?
		COLOR_FLASHING_WHITE : COLOR_FLASHING_BLACK;
}

/* -------------------------------------------------------------------------
    c_initialize_font():     Initialize ROM character table to primary char set
   ------------------------------------------------------------------------- */
void c_initialize_font()
{
    int			c, i, j, v, p;

    for (p = 0, c = 0; c < 256; c++)
	for (i = c * 8; i < c * 8 + 8; i++, p++)
	    for (v = char_rom[ i ] >> 2, j = 0; j < 7; j++, v >>= 1, p++)
	        expanded_font[ p ] = (c < 128) ?
					  ((v & 1) ? 0 :
					   COLOR_LIGHT_WHITE) :
					  ((v & 1) ?
					   COLOR_LIGHT_WHITE : 0);

#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	c_set_iie_cursor();
    }

    if (altchar_flag)
	c_set_altchar();
    else
#endif
	c_set_primary_char();
/*     for (c = 0x40, p = c * 64; c < 0x80; c++) */
/*         for (i = c * 8; i < c * 8 + 64; i++, p++) */
/* 	    expanded_font[ p ] = (expanded_font[ p ] == COLOR_LIGHT_WHITE) ? */
/* 		COLOR_FLASHING_WHITE : COLOR_FLASHING_BLACK; */
}


/* -------------------------------------------------------------------------
    c_initialize_row_col_tables()
   ------------------------------------------------------------------------- */
void c_initialize_row_col_tables()
{
    int x, y, off, i;

    /* text page offsets */
    for (y = 0; y < 24; y++)
	for (x = 0; x < 40; x++)
	    text_page_rows[ video_line_offset[ y ] + x ] = y,
	    text_page_cols[ video_line_offset[ y ] + x ] = x;

    /* hires page offsets. initialize to invalid values. */
    for (i = 0; i < 8 K; i++) {
	hires_page_offset[i] = 65000;
    }

    for (y = 0; y < 24; y++)
	for (off = 0; off < 8; off++)
	    for (x = 0; x < 40; x++) {
		hires_page_offset[video_line_offset[y] + 0x400*off + x ]
		    = (y*8 + off + 4) * 320 + x*7 + 20;
		hires_col_offset[video_line_offset[y]+ 0x400*off + x] =
		    (unsigned char)x;
	    }

#ifdef APPLE_IIE
    /* dhires page offsets */
    for (y = 0; y < 24; y++)
	for (off = 0; off < 8; off++)
	    for (x = 0; x < 40; x++)
		dhires_aux_page_offset[video_line_offset[y] + 0x400*off + x]
		    = (y*8 + off + 4) * 640 + x*7*2 + 40;
#endif
}

/* -------------------------------------------------------------------------
    c_initialize_colors():    Initialize color palette
   ------------------------------------------------------------------------- */

void c_initialize_colors()
{
/*     static int col[ 32 ] = { 0, 6, 10, 14, 18, 22, 25, 28, */
/* 			     31, 34, 37, 39, 41, 43, 45, 47, */
/* 			     48, 49, 50, 51, 52, 53, 54, 55, */
/* 			     56, 57, 58, 59, 60, 61, 62, 63 }; */

    static int col2[ 3 ] = { 27, 40, 62 };

    int				i, j;

    /* HACK - is this necessary?!!!!!!!!!!!!!*/
/*     for (i = 0; i < 8; i++) */
/* 	for (j = 0; j < 32; j++) */
/* 	    vga_setpalette( j+i*32, (i & 1) ? col[ j ] : 0, */
/*                                     (i & 2) ? col[ j ] : 0, */
/*                                     (i & 4) ? col[ j ] : 0 ); */

    /* align the palette for hires graphics */
    for (i = 0; i < 8; i++)
	for (j = 0; j < 3; j++)
	    vga_setpalette( j+i*3+32, (i & 1) ? col2[ j ] : 0,
                                      (i & 2) ? col2[ j ] : 0,
                                      (i & 4) ? col2[ j ] : 0 );

    vga_setpalette( COLOR_FLASHING_BLACK, 0, 0, 0 );
    vga_setpalette( COLOR_FLASHING_WHITE, 63, 63, 63 );
    vga_setpalette( COLOR_FLASHING_GREEN, 0, 63, 0 );
    vga_setpalette( COLOR_FLASHING_UNGREEN, 0, 0, 0 );

#if 0
    /* lores colors */
    vga_setpalette( 0, 0, 0, 0 );		/* Black		*/
    vga_setpalette( 1, 255, 0, 255 ); 		/* Magenta 		*/
    vga_setpalette( 2, 0, 0, 255 );		/* Medium blue		*/
    vga_setpalette( 3, 0, 32, 0 );		/* Purple		*/
    vga_setpalette( 4, 160, 32, 240 );		/* Dark green		*/
    vga_setpalette( 5, 127, 127, 127 );		/* Gray	(50%)		*/
    vga_setpalette( 6, 30, 144, 255 );		/* Dodger blue		*/
    vga_setpalette( 7, 173, 216, 230 );		/* Light blue		*/
    vga_setpalette( 8, 165, 42, 42 );		/* Brown		*/
    vga_setpalette( 9, 255, 69, 0 );		/* Orange red		*/
    vga_setpalette( 10, 127, 127, 127 );	/* Gray (50%)		*/
    vga_setpalette( 11, 255, 192, 203 );	/* Pink			*/
    vga_setpalette( 12, 0, 255, 0 );		/* Green		*/
    vga_setpalette( 13, 255, 255, 0 );		/* Yellow		*/
    vga_setpalette( 14, 127, 255, 212 );	/* Aquamarine		*/
    vga_setpalette( 15, 255, 255, 255 );	/* White		*/
#endif

    /* dhires colors are bass-ackwards because it's easier for me to
       deal with the endianness. */
    vga_setpalette( 0x0, 0, 0, 0 ); 		/* Black 		*/
    vga_setpalette( 0x1, 0, 0, 160 );		/* Magenta <med blue>	*/
    vga_setpalette( 0x2, 0, 100, 0 );		/* Brown <dark green>	*/
    vga_setpalette( 0x3, 28, 134, 255 );	/* Orange <dodger blue>	*/
    vga_setpalette( 0x4, 165, 21, 10 );		/* Dark Green <brown>	*/
    vga_setpalette( 0x5, 165, 42, 42);		/* Gray2 <light gray>	*/
    vga_setpalette( 0x6, 0, 255, 0 );		/* Green <Green>	*/
    vga_setpalette( 0x7, 160, 255, 160 );	/* Yellow <Aqua>	*/
    vga_setpalette( 0x8, 112, 192, 140 );	/* Dark Blue <magenta>	*/
    vga_setpalette( 0x9, 105, 205, 170 );	/* Purple <Purple>	*/
    vga_setpalette( 0xa, 90, 90, 90 );		/* Gray1 <dark gray>	*/
    vga_setpalette( 0xb, 131, 111, 250 );	/* Pink <slateblue>	*/
    vga_setpalette( 0xc, 255, 198, 203 );	/* Medium Blue <Orange>	*/
    vga_setpalette( 0xd, 255, 167, 165 );	/* Light Blue <Pink>	*/
    vga_setpalette( 0xe, 255, 255, 0 );		/* Aqua <Yellow>	*/
    vga_setpalette( 0xf, 255, 255, 255 );	/* White		*/
}

/* -------------------------------------------------------------------------
    c_initialize_code_tables()
   ------------------------------------------------------------------------- */

void c_initialize_code_tables()
{
    int		i;

    for (i = 0; i < 256; i++)
    {
	unsigned char	val = 0;

	if (i & C_Flag)
	    val |= C_Flag_6502;
	if (i & X_Flag)
	    val |= X_Flag_6502;
	if (i & I_Flag)
	    val |= I_Flag_6502;
	if (i & V_Flag)
	    val |= V_Flag_6502;
	if (i & B_Flag)
	    val |= B_Flag_6502;
	if (i & D_Flag)
	    val |= D_Flag_6502;
	if (i & Z_Flag)
	    val |= Z_Flag_6502;
	if (i & N_Flag)
	    val |= N_Flag_6502;

	table_encode_flags[ i ] = val | 0x20;
	table_decode_flags[ val ] = i;
    }
}


/* -------------------------------------------------------------------------
    c_initialize_tables()
   ------------------------------------------------------------------------- */

void c_initialize_tables() {

    int i, x, y;

    /* reset everything */
    for (i = 0; i < 0x10000; i++) {
#ifdef APPLE_IIE
	if (apple_mode == IIE_MODE) {
	    table_read_memory[i] = iie_read_ram_default;
	    table_write_memory[i] = iie_write_ram_default;
	} else
#endif
        {
	    table_read_memory[i] = read_ram_default;
	    table_write_memory[i] = write_ram_default;
	}
    }

    /* basic softswich rom */
    for (i = 0xC000; i < 0xD000; i++)
	table_write_memory[i] =
	    ram_nop;
#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	/* don't add offsets for softswitch rom */
	for (i = 0xC000; i < 0xC100; i++)
	    table_read_memory[i] =
		read_ram_default;
    }
#endif

    /* language card read/write area */
    for (i = 0xD000; i < 0xE000; i++) {
#ifdef APPLE_IIE
	if (apple_mode == 2) {
	    table_write_memory[i] =
		iie_write_ram_bank;
	    table_read_memory[i] =
		iie_read_ram_bank;
	} else
#endif
	{
	    table_write_memory[i] =
		write_ram_bank;
	    table_read_memory[i] =
		read_ram_bank;
	}
    }
    for (i = 0xE000; i < 0x10000; i++) {
#ifdef APPLE_IIE
	if (apple_mode ==2) {
	    table_write_memory[i] =
		iie_write_ram_lc;
	    table_read_memory[i] =
		iie_read_ram_lc;
	} else
#endif
	{
	    table_write_memory[i] =
		write_ram_lc;
	    table_read_memory[i] =
		read_ram_lc;
	}
    }
    /* done common initialization */

#ifdef APPLE_IIE
    /* initialize zero-page, //e specific */
    if (apple_mode == IIE_MODE) {
	for (i = 0; i < 0x200; i++) {
	    table_read_memory[i] =
		iie_read_ram_zpage_and_stack;
	    table_write_memory[i] =
		iie_write_ram_zpage_and_stack;
	}
    }
#endif

#ifdef APPLE_IIE
    /* initialize screen holes (page 1 only). page1 screen holes are
       switched with 80STORE.  page 2 screen holes are switched with
       regular RAMRD/RAMWRT. */
    for (i = 0x400; i < 0x800; i++) {
	table_read_memory[i] =
	    iie_read_screen_hole_text_page0;
	table_write_memory[i] =
	    iie_write_screen_hole_text_page0;
    }
    for (i = 0x2000; i < 0x4000; i++) {
	table_read_memory[i] =
	    iie_read_screen_hole_hires_page0;
	table_write_memory[i] =
	    iie_write_screen_hole_hires_page0;
    }
#endif

    /* initialize text/lores & hires graphics */
    for (y = 0; y < 24; y++) {		/* 24 rows */
	for (x = 0; x < 40; x++)	/* 40 cols */
	{
#ifdef APPLE_IIE
	    if (apple_mode == IIE_MODE) {
		/* //e mode: text/lores page 0 */
		table_write_memory[ video_line_offset[ y ] + x + 0x400] =
			(y < 20) ? iie_write_ram_text_page0 :
				   iie_write_ram_text_mixed0;
		table_read_memory[ video_line_offset[ y ] + x + 0x400] =
		    iie_read_ram_text_page0;
	    }
	    else
#endif
	    {
		/* ][+ modes: text/lores page 0 */
		table_write_memory[ video_line_offset[ y ] + x + 0x400] =
			(y < 20) ? write_ram_text_page0 :
				   write_ram_text_mixed0;
	    }

#ifdef APPLE_IIE
	    if (apple_mode == IIE_MODE) {
		table_write_memory[ video_line_offset[ y ] + x + 0x800] =
		    (y < 20) ? iie_write_ram_text_page1 :
		               iie_write_ram_text_mixed1;
	    }
	    else
#endif
	    {
	    /* ][+ modes: text/lores page 1 in main memory */
	    table_write_memory[ video_line_offset[ y ] + x + 0x800] =
			(y < 20) ? write_ram_text_page1 :
				   write_ram_text_mixed1;
	    }

	    for (i = 0; i < 8; i++)
	    {
		/* //e mode: hires/double hires page 0 */
#ifdef APPLE_IIE
		if (apple_mode == IIE_MODE) {
		    table_write_memory[ 0x2000 + video_line_offset[ y ]
					 + 0x400 * i + x ] =
		        (y < 20) ? ((x & 1) ? iie_write_ram_hires_page0_odd :
					      iie_write_ram_hires_page0_even)
				 : ((x & 1) ? iie_write_ram_hires_mixed0_odd :
					      iie_write_ram_hires_mixed0_even);
		    table_read_memory[ 0x2000 + video_line_offset[ y ]
					 + 0x400 * i + x ] =
			iie_read_ram_hires_page0;
		}

		/* ][+ modes: hires page 0 */
		else
#endif
		{
		    table_write_memory[ 0x2000 + video_line_offset[ y ]
					 + 0x400 * i + x ] =
		        (y < 20) ? ((x & 1) ? write_ram_hires_page0_odd :
					      write_ram_hires_page0_even)
				 : ((x & 1) ? write_ram_hires_mixed0_odd :
					      write_ram_hires_mixed0_even);
		}

#ifdef APPLE_IIE
		if (apple_mode == IIE_MODE) {
		    table_write_memory[ 0x4000 + video_line_offset[ y ]
					 + 0x400 * i + x ] =
		        (y < 20) ? ((x & 1) ? iie_write_ram_hires_page1_odd :
					      iie_write_ram_hires_page1_even)
				 : ((x & 1) ? iie_write_ram_hires_mixed1_odd :
					      iie_write_ram_hires_mixed1_even);
		}

		/* ][+ modes: hires page 1 */
		else
#endif
		{
		table_write_memory[ 0x4000 + video_line_offset[ y ]
				        + 0x400 * i + x ] =
		        (y < 20) ? ((x & 1) ? write_ram_hires_page1_odd :
					      write_ram_hires_page1_even)
				 : ((x & 1) ? write_ram_hires_mixed1_odd :
					      write_ram_hires_mixed1_even);
		}
	    }
	}
    }

    /* keyboard data and strobe (READ) */
    for (i = 0xC000; i < 0xC010; i++) {
	table_read_memory[i] =
	    (Function) read_keyboard;
    }
    /* initialize base area */
    for (i = 0xC000; i < 0xC010; i++) {
	table_write_memory[i] =
	    ram_nop;
    }
    for (i = 0xC010; i < 0xC020; i++) {
	table_read_memory[i] =
	table_write_memory[i] =
	    c_read_keyboard_strobe;
    }

#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {

	/* RDBNK2 switch */
	table_read_memory[0xC011] =
	    iie_check_bank;

	/* RDLCRAM switch */
	table_read_memory[0xC012] =
	    iie_check_lcram;

	/* 80STORE switch */
	table_write_memory[0xC000] = iie_80store_off;
	table_write_memory[0xC001] = iie_80store_on;
	table_read_memory[0xC018] = iie_check_80store;

	/* RAMRD switch */
	table_write_memory[0xC002] = iie_ramrd_main;
	table_write_memory[0xC003] = iie_ramrd_aux;
	table_read_memory[0xC013] = iie_check_ramrd;

	/* RAMWRT switch */
	table_write_memory[0xC004] = iie_ramwrt_main;
	table_write_memory[0xC005] = iie_ramwrt_aux;
	table_read_memory[0xC014] = iie_check_ramwrt;

	/* ALTZP switch */
	table_write_memory[0xC008] = iie_altzp_main;
	table_write_memory[0xC009] = iie_altzp_aux;
	table_read_memory[0xC016] = iie_check_altzp;

	/* 80COL switch */
	table_write_memory[0xC00C] = iie_80col_off;
	table_write_memory[0xC00D] = iie_80col_on;
	table_read_memory[0xC01F] = iie_check_80col;

	/* ALTCHAR switch */
	table_write_memory[0xC00E] = iie_altchar_off;
	table_write_memory[0xC00F] = iie_altchar_on;
	table_read_memory[0xC01E] = iie_check_altchar;

	/* SLOTC3ROM switch */
	table_write_memory[0xC00A] = iie_c3rom_internal;
	table_write_memory[0xC00B] = iie_c3rom_peripheral;
	table_read_memory[0xC017] = iie_check_c3rom;

	/* SLOTCXROM switch */
	table_write_memory[0xC006] = iie_cxrom_peripheral;
	table_write_memory[0xC007] = iie_cxrom_internal;
	table_read_memory[0xC015] = iie_check_cxrom;

	/* RDVBLBAR switch */
	table_read_memory[0xC019] =
	    iie_check_vbl;
    }
#endif
    
    /* random number generator */
    for (i = 0xC020; i < 0xC030; i++)
	table_read_memory[i] =
	table_write_memory[i] =
	    read_random;

    /* TEXT switch */
    table_read_memory[0xC050] =
    table_write_memory[0xC050] =
	read_switch_graphics;
    table_read_memory[0xC051] =
    table_write_memory[0xC051] =
	read_switch_text;

#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC01A] =
	    iie_check_text;
    }
#endif

    /* MIXED switch */
    table_read_memory[0xC052] =
    table_write_memory[0xC052] =
	read_switch_no_mixed;
    table_read_memory[0xC053] =
    table_write_memory[0xC053] =
	read_switch_mixed;

#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC01B] =
	    iie_check_mixed;
    }
    /* PAGE2 switch */
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC054] =
	table_write_memory[0xC054] =
	    iie_page2_off;
    } else
#endif
    {
	table_read_memory[0xC054] =
	table_write_memory[0xC054] =
	    read_switch_primary_page;
    }

#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC01C] =
	    iie_check_page2;
    }
    /* PAGE2 switch */
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC055] =
	table_write_memory[0xC055] =
	    iie_page2_on;
    } else
#endif
    {
	table_read_memory[0xC055] =
	table_write_memory[0xC055] =
	    read_switch_secondary_page;
    }

    /* HIRES switch */
#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC01D] =
	    iie_check_hires;
	table_read_memory[0xC056] =
	table_write_memory[0xC056] =
	    iie_hires_off;
	table_read_memory[0xC057] =
	table_write_memory[0xC057] =
	    iie_hires_on;
    }
    else
#endif
    {
	table_read_memory[0xC056] =
	table_write_memory[0xC056] =
	    read_switch_lores;
	table_read_memory[0xC057] =
	table_write_memory[0xC057] =
	    read_switch_hires;
    }

    /* game I/O switches */
    table_read_memory[0xC061] =
    table_read_memory[0xC069] =
        read_button0;
    table_read_memory[0xC062] =
    table_read_memory[0xC06A] =
        read_button1;
    table_read_memory[0xC063] =
    table_read_memory[0xC06B] =
        read_button2;
    table_read_memory[0xC064] =
    table_read_memory[0xC06C] =
        read_gc0;
    table_read_memory[0xC065] =
    table_read_memory[0xC06D] =
        read_gc1;
#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC066] =
	    iie_read_gc2;
	table_read_memory[0xC067] =
	    iie_read_gc3;
    }
#endif
    for (i = 0xC070; i < 0xC080; i++)
	table_read_memory[i] =
	table_write_memory[i] =
	    read_gc_strobe;

#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	/* IOUDIS switch & read_gc_strobe */
	table_write_memory[0xC07E] =
	    iie_ioudis_on;
	table_write_memory[0xC07F] =
	    iie_ioudis_off;
	table_read_memory[0xC07E] =
	    iie_check_ioudis;
	table_read_memory[0xC07F] =
	    iie_check_dhires;

	/* DHIRES/Annunciator switches */
	table_write_memory[0xC05E] =
	table_read_memory[0xC05E] =
	    iie_dhires_on;
	table_write_memory[0xC05F] =
	table_read_memory[0xC05F] =
	    iie_dhires_off;
    }
#endif

    /* language card softswitches */
    table_read_memory[0xC080] = table_write_memory[0xC080] =
    table_read_memory[0xC084] = table_write_memory[0xC084] =
	lc_c080;
    table_read_memory[0xC081] = table_write_memory[0xC081] =
    table_read_memory[0xC085] = table_write_memory[0xC085] =
	lc_c081;
    table_read_memory[0xC082] = table_write_memory[0xC082] =
    table_read_memory[0xC086] = table_write_memory[0xC086] =
	lc_c082;
    table_read_memory[0xC083] = table_write_memory[0xC083] =
    table_read_memory[0xC087] = table_write_memory[0xC087] =
	lc_c083;

    table_read_memory[0xC088] = table_write_memory[0xC088] =
    table_read_memory[0xC08C] = table_write_memory[0xC08C] =
	lc_c088;
    table_read_memory[0xC089 ] = table_write_memory[0xC089] =
    table_read_memory[0xC08D ] = table_write_memory[0xC08D] =
	lc_c089;
    table_read_memory[0xC08A] = table_write_memory[0xC08A] =
    table_read_memory[0xC08E] = table_write_memory[0xC08E] =
	lc_c08a;
    table_read_memory[0xC08B] = table_write_memory[0xC08B] =
    table_read_memory[0xC08F] = table_write_memory[0xC08F] =
	lc_c08b;

    /* disk softswitches */
    table_read_memory[0xC0E0] = table_read_memory[0xC0E2] =
    table_read_memory[0xC0E4] = table_read_memory[0xC0E6] =
        disk_read_nop;

    table_read_memory[0xC0E1] = table_read_memory[0xC0E3] =
    table_read_memory[0xC0E5] = table_read_memory[0xC0E7] =
        disk_read_phase;

    table_read_memory[0xC0E8] =
	disk_read_motor_off;
    table_read_memory[0xC0E9] =
	disk_read_motor_on;
    table_read_memory[0xC0EA] =
	disk_read_select_a;
    table_read_memory[0xC0EB] =
	disk_read_select_b;
    table_read_memory[0xC0EC] =
	disk_read_byte;
    table_read_memory[0xC0ED] =
	disk_read_latch;	/* read latch */
    table_read_memory[0xC0EE] =
	disk_read_prepare_in;
    table_read_memory[0xC0EF] =
	disk_read_prepare_out;

    for (i = 0xC0E0; i < 0xC0F0; i++) {
        table_write_memory[i] =
	table_read_memory[i];
    }

    table_write_memory[0xC0ED] =
	disk_write_latch;	/* write latch */

#ifdef APPLE_IIE
    /* slot i/o area */
    for (i = 0xC100; i < 0xC300; i++) {
	table_read_memory[i] =
	    iie_read_slotx;		/* slots 1 & 2 (x) */
    }
    for (i = 0xC300; i < 0xC400; i++) {
	table_read_memory[i] =
	    iie_read_slot3;		/* slot 3 (80col) */
    }
    for (i = 0xC400; i < 0xC600; i++) {
	table_read_memory[i] =
	    iie_read_slotx;		/* slots 4 & 5 (x) */
    }
    for (i = 0xC600; i < 0xC700; i++) {
	table_read_memory[i] =
	    iie_read_slot6;		/* slot 6 (disk) */
    }
    for (i = 0xC700; i < 0xC800; i++) {
	table_read_memory[i] =
	    iie_read_slotx;		/* slot 7 (x) */
    }
    for (i = 0xC800; i < 0xD000; i++) {	/* HACK!!!! does it also include CFFF*/
	table_read_memory[i] =
	    iie_read_slot_expansion;	/* expansion rom */
    }
    table_read_memory[0xCFFF] =
    table_write_memory[0xCFFF] =
	iie_disable_slot_expansion;
#endif
}

/* -------------------------------------------------------------------------
    c_initialize_apple_ii_memory()
   ------------------------------------------------------------------------- */

void c_initialize_apple_ii_memory()
{
     FILE	*f;
     int	i;
     static int ii_rom_loaded = 0;
     static int slot6_rom_loaded = 0;
#ifdef APPLE_IIE
     static int iie_rom_loaded = 0;
#endif

     for (i = 0; i < 64 K; i++) {
         apple_ii_64k[0][i] = 0;
         apple_ii_64k[1][i] = 0;
     }
     for (i = 0; i < 8 K; i++)
         language_card[0][i] = language_card[1][i] = 0;
     for (i = 0; i < 8 K; i++)
         language_banks[0][i] = language_banks[1][i] = 0;

     if (!ii_rom_loaded)
     {
         sprintf(temp, "%s/apple_II.rom", system_path);
         if ((f = fopen(temp, "r")) == NULL) {
             printf("Cannot find file '%s'.\n",temp);
	     exit(0);
         }
         fread(apple_ii_rom, 0x3000, 1, f);
         fclose(f);
	 ii_rom_loaded = 1;
     }

#ifdef APPLE_IIE
     if (!iie_rom_loaded) {
	 sprintf(temp, "%s/apple_IIe.rom", system_path);
	 if ((f = fopen(temp, "r")) == NULL) {
	     printf("Cannot find file '%s'.\n",temp);
	     exit(0);
	 }
	 fread(apple_iie_rom, 32 K, 1, f);
	 fclose(f);
	 iie_rom_loaded = 1;
     }
#endif

     for (i = 0xD000; i < 0x10000; i++)
	 apple_ii_64k[0][i] = apple_ii_rom[i - 0xD000];
     for (i = 0; i < 0x1000; i++)
	 language_banks[0][i] = apple_ii_rom[i];
     for (i = 0; i < 0x2000; i++)
	 language_card[0][i] = apple_ii_rom[i + 0x1000];

#ifdef APPLE_IIE
     if (apple_mode == IIE_MODE) {
	 /* load the rom from 0xC000, slot rom main, internal rom aux */
	 for (i = 0xC000; i < 0x10000; i++) {
	     apple_ii_64k[0][i] = apple_iie_rom[i - 0xC000];
	     apple_ii_64k[1][i] = apple_iie_rom[i - 0x8000];
/* 	     apple_ii_64k[1][i] = apple_iie_rom[i - 0xC000]; */
/* 	     apple_ii_64k[0][i] = apple_iie_rom[i - 0x8000]; */
	 }
	 for (i = 0; i < 0x1000; i++) {
	     language_banks[0][i] = apple_iie_rom[i + 0x1000];
	     language_banks[1][i] = apple_iie_rom[i + 0x5000];
/* 	     language_banks[1][i] = apple_iie_rom[i + 0x1000]; */
/* 	     language_banks[0][i] = apple_iie_rom[i + 0x5000]; */
	 }
	 for (i = 0; i < 0x2000; i++) {
	     language_card[0][i] = apple_iie_rom[i + 0x2000];
	     language_card[1][i] = apple_iie_rom[i + 0x6000];
/* 	     language_card[1][i] = apple_iie_rom[i + 0x2000]; */
/* 	     language_card[0][i] = apple_iie_rom[i + 0x6000]; */
	 }
     }
     else
#endif
     /* softswitch memory HACK - why this? */
     {
	 for (i = 0xC100; i < 0xD000; i++) {
	     apple_ii_64k[0][i] = i & 0xFF;
	     apple_ii_64k[1][i] = i & 0xFF;
	 }
     }

     /* HACK - why this? */
     apple_ii_64k[0][0xC000] = 0x00;
     apple_ii_64k[1][0xC000] = 0x00;

#ifdef APPLE_IIE		/* HACK!! */
     if (apple_mode == IIE_MODE) {
     } else
#endif
     {
     /* load Disk II rom */
     if (!slot6_rom_loaded) {
	 sprintf(temp, "%s/slot6.rom", system_path);
	 if ((f = fopen( temp, "r" )) == NULL) {
	     printf("Cannot find file '%s'.\n",temp);
	     exit( 0 );
	 }
	 fread(slot6_rom, 0x100, 1, f);
	 fclose(f);
	 slot6_rom_loaded = 1;
     }
     memcpy(apple_ii_64k[0] + 0xC600, slot6_rom, 0x100);
     memcpy(apple_ii_64k[1] + 0xC600, slot6_rom, 0x100);	/* slot rom */
     }
}

/* -------------------------------------------------------------------------
    void c_initialize_sound()
   ------------------------------------------------------------------------- */

void c_initialize_sound()
{
    int i;

    ioperm( 0x42, 1, 1 );
    ioperm( 0x61, 1, 1 );

    /* HACK - check errno */

    for (i = 0xC030; i < 0xC040; i++)
        table_read_memory[i] = table_write_memory[i] =
	    sound_mode ? read_speaker_toggle_pc : ram_nop;
}

#ifdef APPLE_IIE
/* -------------------------------------------------------------------------
    c_initialize_iie_switches
   ------------------------------------------------------------------------- */
void c_initialize_iie_switches() {

    lc_offset = 0;
    zp_offset = 0;
    ramrd_offset = 0;
    ramwrt_offset = 0;
    write_text_page_offset = 0;
    read_text_page_offset = 0;
    write_hires_page_offset = 0;
    read_hires_page_offset = 0;
/*     page2_setting = 0; */

    vmode_dhires = 0;
    dhires_flag = 0;
    eighty_store_flag = 0;
    ramrd_flag = 0;			/* MAIN > 127 */
    ramwrt_flag = 0;			/* MAIN > 127 */
    altzp_flag = 0;
    eighty_col_flag = 0;
    altchar_flag = 0;
    ioudis_flag = 0x80;			/* ioudis on */

    c3rom_flag = 0x80;/*HACK0or1?*/	/* c3rom internal */
    c8rom_flag = 0x80;			/* c8rom internal - HACK!! */
    cxrom_flag = 0;			/* cxrom peripheral */

    c3rom_offset = 0x10000;		/* c3rom internal */
    c8rom_offset = 0x10000;		/* c8rom internal - HACK!! */
    cxrom_offset = 0;			/* cxrom peripheral */
}
#endif

/* -------------------------------------------------------------------------
    c_initialize_lc_settings()
   ------------------------------------------------------------------------- */
void c_initialize_lc_settings() {
    language_card_read = 0;		/* READ ROM */
    language_card_write = 1;		/* WRITE TO LC RAM */
    language_card_second = 1;		/* because we're writing */
    language_current_bank = 0;
}

/* -------------------------------------------------------------------------
    void c_initialize_vm()
   ------------------------------------------------------------------------- */
void c_initialize_vm() {
    c_initialize_font();		/* font already read in */
    c_initialize_hires_values();	/* precalculate hires values */
    c_initialize_row_col_tables();	/* precalculate hires offsets */
    c_initialize_code_tables();		/* processor flags */
    c_initialize_tables();		/* read/write memory jump tables */
    c_initialize_apple_ii_memory();	/* read in rom memory */
    c_initialize_sound();		/* sound system */
    c_init_6();				/* drive ][, slot 6 */

    /* initial language card settings HACK - depends on apple_mode */
    c_initialize_lc_settings();		/* language card settings */
#ifdef APPLE_IIE
    c_initialize_iie_switches();	/* set the //e softswitches */
    c_initialize_dhires_values();	/* set up dhires colors */
#endif

    c_initialize_mouse();
}

/* -------------------------------------------------------------------------
    void c_initialize()
   ------------------------------------------------------------------------- */

#define my_pixel(x, y, c) GM[(y)*320+(x)]=(c)

void c_initialize()
{
    int		x, y;
    vga_modeinfo *modeinfo;

    /* vga_disabledriverreport(); */

    /* HACK - I'm forcing VGA mode since I've never tested any of the
       new stuff with a supported SVGA card.  And, the emulator is
       optimized enough that SVGA isn't going to make it any
       better. uncomment the hackaround to experiment...  -ASC v005
       Feb '98 */
    force_vga_mode = 1;
    if (force_vga_mode) {
	printf("Note: Forcing to standard VGA mode, "
	       "see code for explanation.\n");
	chipset = VGA;
	normal_vga_mode = True;
    } else {
	chipset = vga_getcurrentchipset();
	modeinfo = vga_getmodeinfo(G320x200x256);	
	page_2_offset = 0x10000;

	if (chipset == UNDEFINED) {
	    vga_setmode(TEXT);
	    printf("SVGAlib couldn't detect an appropriate chipset.\n");
	    exit( 0 );
	}
	
	/* Trident cards use a different scan line width */
	if (chipset == TVGA8900)
	    page_2_offset /= 4;
	
	if (page_2_offset > modeinfo->startaddressrange) {
	    printf("Warning: SVGAlib cannot use SVGA features on this chipset.\n"
		   "         Standard VGA is used instead with some performance "
		   "degradation.\n");
	    normal_vga_mode = True;
	}
    }
    
    if (normal_vga_mode) {
	printf("Press RETURN to continue...");
	getchar();
    }

    /* read in system files and calculate system defaults */
    c_read_font_file();
    c_load_interface_font();

    /* set the state of the virtual machine */
    c_initialize_vm();

    vga_setchipset( chipset );
    vga_setmode( G320x200x256 );
    vga_claimvideomemory( 131072 );	/* 2 screens */
    c_initialize_keyboard();

    svga_GM = GM = vga_getgraphmem();
    c_initialize_colors();

    c_setpage( 1 );

    for (y = 0; y < 200; y++)
	for (x = 0; x < 320; x++)
            my_pixel( x, y, 0 );

    c_setpage( 0 );

    for (y = 0; y < 200; y++)
	for (x = 0; x < 320; x++)
            my_pixel( x, y, 0 );

    vmode_text =/* vmode_mixed =*/ True;
    vmode_mixed = False;
#ifdef APPLE_IIE
    vmode_screen =
#endif
    vmode_page2 = vmode_hires = False;
    vmode_active = False;
}

void c_read_random() {
   random_value = (unsigned char)(rand() >> 8);
}

void c_system_defaults() {

    FILE *f = NULL;

    seteuid(user);		/* become user */
    sprintf(temp, "%s/%s", getenv("HOME"), ".apple2");
    f = fopen(temp, "r");
    if (f == NULL) {
	printf(
	    "Warning. Cannot open the .apple2 system defaults file.\n"
	    "Make sure it's readable in your home directory.");
	printf("Press RETURN to continue...");
	getchar();
	return;
    }

    line_num = 1;
    if (f) {
	prefsin = f;
	prefslex();		/* call the lexical scanner */
    }
    fclose( f );
    seteuid(privileged);	/* become privileged */
}

/* save user prefs */
void c_save_settings() {
    FILE *f = NULL;

    seteuid(user);		/* become user */
    sprintf(temp, "%s/%s", getenv("HOME"), ".apple2");
    f = fopen(temp, "w");
    if (f == NULL) {
	printf(
	    "Cannot open the .apple2 system defaults file for writing.\n"
	    "Make sure it has rw permission in your home directory.");
	return;
    }
    
    fprintf(f, "speed = %d\n", MAX_APPLE_DELAY + 1 - apple_speed);
    fprintf(f, "mode = %s\n",  (apple_mode == 0) ? "][+" :
			       (apple_mode == 1) ? "][+ undocumented" :
						   "//e");
    fprintf(f, "disk path = %s\n", disk_path);
    fprintf(f, "color = %s\n", (color_mode == 0) ? "black/white" :
			       (color_mode == 1) ? "lazy color" :
			       (color_mode == 2) ? "color" :
			       (color_mode == 3) ? "lazy interpolated" :
						   "interpolated");
    fprintf(f, "sound = %s\n", (sound_mode == 0) ? "off" :
						   "pc speaker");
    fprintf(f, "joystick = %s\n", (joy_mode == JOY_KYBD)    ? "keyboard" :
				  (joy_mode == JOY_DIGITAL) ? "digital" :
				  (joy_mode == JOY_PCJOY)   ? "pc joystick" :
							      "off");
    fprintf(f, "joystick range = %d\n", joy_range);
    fprintf(f, "origin_x = %d\n", joy_center_x);
    fprintf(f, "origin_y = %d\n", joy_center_y);
#ifdef PC_JOYSTICK
    fprintf(f, "pc joystick parms = %d %d %d %d %d %d %ld\n",
	    js_center_x, js_center_y, js_max_x, js_min_x,
	    js_max_y, js_min_y, js_timelimit);
#endif
    fprintf(f, "sensitivity = %d%%\n", joy_step);
    fprintf(f, "system path = %s\n", system_path);
    fclose(f);
    seteuid(privileged);	/* become privileged */
}


int main(int argc, char *argv[])
{
    int		i;
    user = getuid();
    privileged = geteuid();

#ifdef DEBUGGER
    watch_addrs = -1;
    watch_data = -1;
    /*  the watchpoints and breakpoints */
    for (i=0; i<MAX_BRKPTS; i++) {
	breakpoints[i] = -1;
	watchpoints[i] = -1;
    }
    for (i=0; i<0x100; i++) {
	op_breakpoints[(unsigned char)i] = 0;
    }
#endif

    for (i = 1; i < argc; i++)
    {
	lowercase_string(argv[i]);
	if (strcmp(argv[i], "-vga") == 0)
	    force_vga_mode = True;
    }

    c_system_defaults();		/* handle user prefs */
    c_initialize();			/* init svga graphics and vm */

    exception_flag = ResetSig;		/* flag reset state */

    vmode_active = False;
    c_setpage( 0 );
    c_setscreen( 0 );

    for (;;) {
#ifdef APPLE_IIE
	if (apple_mode == IIE_MODE)
	    assign_65c02_table();
	else
#endif
#ifdef II_UNDOCUMENTED
	if (apple_mode == IIU_MODE)
		assign_6502_undoc_table();
	else
#endif
	    assign_6502_table();

	/* execute the emulator */
        cpu65x02();

#ifdef DEBUGGER
	/*  the watchpoints and breakpoints */
	for (i=0; i<MAX_BRKPTS; i++) {
	    breakpoints[i] = -1;
	    watchpoints[i] = -1;
	}
	for (i=0; i<0x100; i++) {
	    op_breakpoints[(unsigned char)i] = 0;
	}
#endif

	c_initialize_vm();
/* 	c_initialize_tables(); */
/*      c_initialize_apple_ii_memory(); */
/* 	c_initialize_sound(); */
/*      c_init_6(); */

	vmode_text =/* vmode_mixed =*/ True;
	vmode_mixed = False;
#ifdef APPLE_IIE
	vmode_screen =
#endif
        vmode_page2 = vmode_hires = False;

/*      language_card_read = 0;*/		/* HACK - how to reset? */
/* 	language_card_write = 2; */
/* 	language_card_second = 1; */
/*      language_current_bank = 0; */

	exception_flag = ResetSig;

        vmode_active = False;
	c_setpage( 0 );
	c_setscreen( 0 );

	c_update_video_screen();
    }

    /* for safety */
    vga_setmode(TEXT);
    keyboard_close();
    exit( 0 );
}
