/*
 *  desktop -- The 3dfx Desktop Demo 
 *  COPYRIGHT 3DFX INTERACTIVE, INC. 1999
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GXFFILE_H__
#define __GXFFILE_H__


typedef float GXFVector[4];
typedef float GXFVector3[3];

#define SET_GXF_VECTOR(v, x, y, z, w)  (v)[0] = (x); (v)[1] = (y); (v)[2] = (z); (v)[3] = (w);

// version
#define VERS_ID 'SREV'
// map list
#define MAPL_ID 'LPAM'
// material list
#define MATL_ID 'LTAM'
// object list
#define OBJL_ID 'LJBO'
// object node list
#define OBNL_ID 'LNBO'
// node list
#define NDEL_ID 'LEDN'
// animation list
#define ANIL_ID 'LINA'
// camera list
#define CAML_ID 'LMAC'
// light list
#define LITL_ID 'LTIL'

#define FORM_ID 'MROF'
#define GXF_ID  ' FXG'
#define WILD_ID '\?\?\?\?'


// **** MaterialTexture ****
enum MaterialTexture
{
	AMBIENT_TEXTURE = 0,
	DIFFUSE_TEXTURE,
	SPECULAR_TEXTURE,
	SHININESS_TEXTURE,
	SHIN_STRENGTH_TEXTURE,
	SELF_ILLUM_TEXTURE,
	OPACITY_TEXTURE,
	FILTER_COLOR_TEXTURE,
	BUMP_MAP_TEXTURE,
	REFLECTION_TEXTURE,
	REFRACTION_TEXTURE,
	DISPLACEMENT_TEXTURE,
	NUM_MATERIAL_TEXTURES
};

#define HAS_ANIMATION_CHUNK 0x00000001

// **** GXF file data structs ****

typedef struct _gxf_header_
{
	unsigned long form;
	unsigned long file_size;
	unsigned long form_id;
} GXF_Header;

typedef struct _iff_header_
{
	unsigned long chunk_id;
	unsigned long data_size;
} IFF_Header;

typedef struct _iff_chunk_
{
	IFF_Header header;
	unsigned char *data;
} IFF_Chunk;

typedef struct _gxf_vertex_
{
	float x,y,z;
} GXFVertex;

typedef struct _gxf_uv_record_
{
	float s,t;
} GXFUVRecord;


// **** GXFTexMap ****

#define MAX_FNAME 32

typedef struct _gxf_texmap_record_
{
	char texmap_name[MAX_FNAME];
} GXFTexMapRecord;

class GXFTexMap
{

public:
	GXFTexMap(IFF_Chunk *chunk);
	~GXFTexMap();

	int IsValid() { return m_valid;}				// return validity

	int GetNumTexMaps() {return m_num_texmaps;}		// number of texture maps
	char *GetTexMapName(int index) {return (char *) m_texmaps[index]; } // return the file name for the nth texmap

private:

	int m_valid;						// for testing validity of chunk data

	int m_num_texmaps;					// number of frames in scene (this will move shortly)
	
	IFF_Chunk *m_texmap_chunk;			// the pointer to the beginning of the tex map chunk

	GXFTexMapRecord **m_texmaps;		// array of pointers to all of the file names

};


// **** GXFMaterial class ****

typedef struct _gxf_mat_data_record_
{
	int texture_id;
	int wrapping;
	float opacity;
} GXFMatDataRecord;

typedef struct _gxf_material_record_
{
	float diffuse_r,diffuse_g,diffuse_b;
	float specular_r,specular_g,specular_b;
	GXFMatDataRecord mat_data[NUM_MATERIAL_TEXTURES];
} GXFMaterialRecord;

class GXFMaterial
{
	
public:
	GXFMaterial(IFF_Chunk *chunk);
	~GXFMaterial();

	int IsValid() { return m_valid;}				// return validity
	
	int GetNumMaterials() {return m_num_materials;} // number of lights in chunk
	
	void GetDiffuseRGB(float *r, float *g, float*b, int index);		// return the d RGB for the nth material
	void GetSpecularRGB(float *r, float *g, float*b, int index);	// return the s RGB for the nth material
	float GetSpecularR(int index);									// return s R for the ...
	int  GetTextureID(int index, int tex_id)
		{return m_materials[index]->mat_data[tex_id].texture_id;} // return the texture id for nth material's nth texture
	int  GetTextureWrapping(int index, int tex_id)
		{return m_materials[index]->mat_data[tex_id].wrapping;} // return the wrapping for nth material's nth texture
	float  GetTextureOpacity(int index, int tex_id)
		{return m_materials[index]->mat_data[tex_id].opacity;} // return the opacity for nth material's nth texture
	
private:
	
	int m_valid;						// for testing validity of chunk data
	
	int m_num_materials;				// number of materials in scene
	
	IFF_Chunk *m_material_chunk;		// the pointer to the beginning of the materials chunk
	
	GXFMaterialRecord **m_materials;	// array of pointers to all of the material records
	
};

// **** GXFObject ****


typedef struct _gxf_triangle_
{
	unsigned int v0,v1,v2;			   // the indices into the vertices
} GXFTriangle;


typedef struct _gxf_mesh_record_hdr_
{
	unsigned int material_id;			// material id
	int num_primitives;					// number of primitives (always 1 right now)
	int primitive_type;					// (always TRIANGLE_LIST right now)
	int num_triangles;
} GXFMeshRecordHdr;

typedef struct _gxf_mesh_record_
{
	GXFMeshRecordHdr *mesh_header;		// mesh record header
	GXFTriangle *triangles;				// pointer to triangles
} GXFMeshRecord;


typedef struct _gxf_object_record_hdr_
{
	unsigned int flags;					// contains number of vertex channels in lo byte
	int num_verts;						// number of vertices
} GXFObjectRecordHdr;


typedef struct _gxf_object_record_
{
	GXFObjectRecordHdr *object_header;	// pointer to start of each object
	int num_tverts;						// number of texture vertices
	int num_meshes;						// number of meshes
	GXFVertex *vertices;				// list of vertices
	GXFVertex *normals;					// list of normals
	GXFUVRecord *t_vertices;			// list of texture vertices
	GXFMeshRecord *meshes;				// list of meshes
} GXFObjectRecord;

class GXFObject
{
	
public:
	GXFObject(IFF_Chunk *chunk);
	~GXFObject();

	int IsValid() { return m_valid;}		// return validity

	int GetNumObjects() { return m_num_objects; } // number of objects

	unsigned int GetFlags(int index);			// get object n's flags

	int GetNumVerts(int index);					// get number of vertices in object n

	int GetNumTVerts(int index);				// get number of texture vertices in object n

	GXFVertex *GetVertexList(int index);		// return list of vertices for n'th object

	GXFVertex *GetNormalList(int index);		// return list of normals for n'th object

	GXFUVRecord *GetTVertexList(int index);		// return list of texture vertices for n'th object

	int GetNumMeshes(int index);				// number of meshes in this object

	int GetMeshMaterialID(int index, int mesh_index);	// material id of nth mesh in nth object
	int GetNumPrimitives(int index, int mesh_index);	// number of primitives in...
	int GetPrimitiveType(int index, int mesh_index, int prim_index); // type of primitives in...
	int GetMeshNumTriangles(int index, int mesh_index); // number of triangles in...

	GXFTriangle *GetMeshTriangleList(int index, int mesh_index); // only 1 primitive so this hacks

private:
	
	int m_valid;						// for testing validity of chunk data
	
	int m_num_objects;					// number of objects in chunk
	
	IFF_Chunk *m_object_chunk;			// the pointer to the beginning of the object chunk
	
	GXFObjectRecord *m_objects;			// array of all of the object records
	
};



// **** GXFObjectNode ****

typedef struct _gxf_object_node_record_
{
	int object_index;
	char name[MAX_FNAME];
	int parent_id;
	int first_child_id;
	int sibling_id;
	int matrix_level;
	float mat[12];				 // matrix
} GXFObjNodeRecord;


class GXFObjectNode
{
	
public:
	GXFObjectNode(IFF_Chunk *chunk);
	~GXFObjectNode();

	int IsValid() { return m_valid;}		// return validity

	int GetNumObjectNodes() { return m_num_object_nodes; }									// number of object nodes
	int GetObjectIndex(int index) { return m_object_nodes[index]->object_index; }			// object index of nth node
	char *GetObjectNodeName(int index) { return m_object_nodes[index]->name; }				// name of nth object node
	int GetObjectParent(int index) { return m_object_nodes[index]->parent_id; }				// parent id of nth node
	int GetObjectFirstChild(int index) { return m_object_nodes[index]->first_child_id; }	// first child id of nth node
	int GetObjectSibling(int index) { return m_object_nodes[index]->sibling_id; }			// sibling index of nth node
	int GetObjectMatrixLevel(int index) { return m_object_nodes[index]->matrix_level; }		// matrix level of nth node
	float GetObjectMatrixElement(int index, int element)
		{ return m_object_nodes[index]->mat[element]; } // return mat[element] of the nth object node
private:
	
	int m_valid;						// for testing validity of chunk data
	
	int m_num_object_nodes;				// number of object nodes in scene
	
	IFF_Chunk *m_object_node_chunk;		// the pointer to the beginning of the object node chunk
	
	GXFObjNodeRecord **m_object_nodes;	// array of pointers to all of the object node records
	
};



// **** GXFAnimation class ****


typedef struct _gxf_animation_record_
{
	float mat[12];

} GXFAnimationRecord;


class GXFAnimation
{

public:
	GXFAnimation(IFF_Chunk *chunk);
	~GXFAnimation();

	int IsValid() { return m_valid;}						// return validity

	int GetNumFrames() {return m_num_frames;}				// number of frames in animation
	int GetNumObjectNodes() {return m_num_object_nodes;}	// number of object nodes

	float GetMatrixElement(int frame, int obj_index, int element); // return mat[element] of the nth object's nth frame

private:

	int m_valid;						// for testing validity of chunk data

	int m_num_frames;					// number of frames

	int m_num_object_nodes;				// number of cameras in scene
	
	IFF_Chunk *m_animation_chunk;		// the pointer to the beginning of the camera chunk

	GXFAnimationRecord **m_frames;		// array of pointers to each frame

};

// **** GXFNode class ****


typedef struct _gxf_node_record_header_
{
	unsigned int flags;					// HAS_ANIMATION_CHUNK is one
	int object_node_index;				// index into the first object node
	int	num_object_nodes;				// how many object nodes are children of this node
	char name[MAX_FNAME];				// name of this node
} GXFNodeRecordHdr;


typedef struct _gxf_node_record_
{
	GXFNodeRecordHdr *node;				// pointer to start of each node
	GXFAnimation *animation;			// GXFAnimation data for this node (NULL if none)
} GXFNodeRecord;

class GXFNode
{
	
public:
	GXFNode(IFF_Chunk *chunk);
	~GXFNode();

	int IsValid() { return m_valid;}		// return validity

	int GetNumNodes() { return m_num_nodes; } // number of nodes

	GXFAnimation *GetGXFAnimation(int index) { return m_nodes[index].animation; } // number of nodes

	int GetNodeFlags(int index) { return m_nodes[index].node->flags; }						// flags of nth node
	int GetObjectNodeIndex(int index) { return m_nodes[index].node->object_node_index; }		// object index of nth node
	int GetNumObjectNodes(int index) { return m_nodes[index].node->num_object_nodes; }					// number of object nodes ownd by nth node
	char *GetNodeName(int index) { return m_nodes[index].node->name; }						// name of nth node

private:
	
	int m_valid;						// for testing validity of chunk data
	
	int m_num_nodes;					// number of nodes in scene
	
	IFF_Chunk *m__node_chunk;			// the pointer to the beginning of the object node chunk
	
	GXFNodeRecord *m_nodes;				// array of all of the node records
	
};


// **** GXFCamera class ****

typedef struct _gxf_camera_frame_
{
	float fov;		// field of view
	float dist;		// distance to target
	GXFVector3 u;	// matrix for rotation
	GXFVector3 v;
	GXFVector3 n;
	GXFVector3 pos;	// position

} GXFCameraFrame;

// **** GXFAnimCamera ****
typedef struct _gxf_anim_camera_
{
	int num_camera_frames;			// number of frames for this animation camera
	GXFCameraFrame *camera_frames;  // start of frame data
} GXFAnimCamera;					


class GXFCamera
{

public:
	GXFCamera(IFF_Chunk *chunk);
	~GXFCamera();

	int IsValid() { return m_valid;}		// return validity

	int GetNumFrames() {return m_num_scene_frames;} // number of frames in scene
	int GetNumCameras() {return m_num_cameras;}		// number of cameras in file

	int GetNumAnimFrames(int index);				// number of animation frames in each camera

	float GetFOV(int index, int frame_num);					// return the FOV for the nth frame of the nth camera
	float GetDistance(int index, int frame_num);			// return the camera distance to target for the...
	void GetU(GXFVector *v_u, int index, int frame_num);	// return the u vector for the...
	void GetV(GXFVector *v_v, int index, int frame_num);	// return the v...
	void GetN(GXFVector *v_n, int index, int frame_num);	// return the n...
	void GetPos(GXFVector *v_p, int index, int frame_num);	// return the pos...

private:

	int m_valid;			// for testing validity of camera data

	int m_num_scene_frames;	// number of frames in scene (this will move shortly)

	int m_num_cameras;		// number of cameras in scene
	
	IFF_Chunk *m_camera_chunk;		// the pointer to the beginning of the camera chunk

	GXFAnimCamera **m_anim_cameras;	// array of pointers to all of the animation cameras

};


// **** GXFLight Class ****

typedef struct _gxf_light_record_
{
	int type;				// light type
	float r,g,b;			// diffuse color
	float atten_start;		// attenuation start
	float atten_end;		// attenuation end
	GXFVector3 u;				// matrix for rotation
	GXFVector3 v;
	GXFVector3 n;
	GXFVector3 pos;			// position

} GXFLightRecord;


class GXFLight
{

public:
	GXFLight(IFF_Chunk *chunk);
	~GXFLight();

	int IsValid() { return m_valid;}		// return validity

	int GetNumLights() {return m_num_lights;} // number of lights in chunk

	int GetLightType(int index);			// return the light type
	void GetDiffuseRGB(float *r, float *g, float*b, int index); // return the RGB for the nth light
	float GetAttenStart(int index);			// return the attenuation start for the nth light
	float GetAttenEnd(int index);			// return the attenuation end for the nth light
	void GetU(GXFVector *v_u, int index);	// return the u vector for the...
	void GetV(GXFVector *v_v, int index);	// return the v...
	void GetN(GXFVector *v_n, int index);	// return the n...
	void GetPos(GXFVector *v_p, int index);	// return the pos...

private:

	int m_valid;					// for testing validity of camera data

	int m_num_lights;				// number of lights in scene
	
	IFF_Chunk *m_lights_chunk;		// the pointer to the beginning of the camera chunk

	GXFLightRecord **m_lights;		// array of pointers to all of the lights

};


class GXFFile
{

public:
	GXFFile(const char *filename);
	~GXFFile();

	int IsValid() { return m_valid; }

	int ParseVersionString(IFF_Chunk *chunk);

	char *GetErrorString() { return m_error_string; }

	char *GetFilePath() { return file_path; }

	GXFTexMap		*GetTexMapInterface() { return gxf_texmaps; }
	GXFMaterial		*GetMaterialInterface() { return gxf_materials; }
	GXFObject		*GetObjectInterface() { return gxf_objects; }
	GXFObjectNode	*GetObjectNodeInterface() { return gxf_object_nodes; }
	GXFNode			*GetNodeInterface() { return gxf_nodes; }
	GXFCamera		*GetCameraInterface() { return gxf_cameras; }
	GXFLight		*GetLightInterface() { return gxf_lights; }

private:

	int				m_valid;

	char			*m_error_string;

	char			file_path[64];

	GXFTexMap		*gxf_texmaps;
	GXFMaterial		*gxf_materials;
	GXFObject		*gxf_objects;
	GXFObjectNode	*gxf_object_nodes;
	GXFNode			*gxf_nodes;
	GXFCamera		*gxf_cameras;
	GXFLight		*gxf_lights;


};

// prototypes

int GetChunk(IFF_Chunk *chunk, unsigned char *data);


#endif // __GXFFILE_H__
