/*
 *  desktop -- The 3dfx Desktop Demo 
 *  COPYRIGHT 3DFX INTERACTIVE, INC. 1999
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <X11/xpm.h>

#include "init.h"

#include "control.h"

/* These are all of the textures that I use for backgrounds */
char *bg_names[NTHUMBS] = {
  "data/Analog-Wind-1",
  "data/Breakfast-Burrito-3",
  "data/Schematic-Diagram-1",
  "data/Hard-Times-Ahead-1",
  "data/Keeblers-Demise-1",
  "data/Life-After-Death-1",
  "data/Blatant-Plug-Insert-1",
  "data/The-Currency-Of-Pain-1",
  "data/Triplets-1",
  "data/JKrinitt"
};

Display *
init_X ()
{
  Display *display;

  /* Connect to the fake X server */
  display = XOpenDisplay (NULL);
  if (display == NULL)
  {
    fprintf (stderr, "Could not open display\n");
    exit (-1);
  }

  /* I need to select input for the root window so that I can catch
   * exposures and update the screen.
   */
  /*XSelectInput (display, RootWindow(display, 0), ExposureMask);*/

  return display;
}

void
init_glide ()
{
  /* Initialize Glide */
  grGlideInit ();

#ifndef USE_GLIDE3
  grSstQueryHardware (&hwconfig);
#endif
  grSstSelect (0);
  /* Passing an HWND of 0x1 turns on the overlay hack */
  grSstWinOpen (0x1, 
		resolution,
		GR_REFRESH_60Hz,
		GR_COLORFORMAT_ABGR,
		GR_ORIGIN_UPPER_LEFT,
		2, 1);  

#ifdef USE_GLIDE3
  /* Define Vertex layout to match the old Glide2x vertex.  This
   * comes right from the Glide3 programming guide. */
  grCoordinateSpace(GR_WINDOW_COORDS); 
  grVertexLayout(GR_PARAM_XY, 0, GR_PARAM_ENABLE); 
  grVertexLayout(GR_PARAM_RGB, 12, GR_PARAM_ENABLE); 
  grVertexLayout(GR_PARAM_Z, 24, GR_PARAM_ENABLE); 
  grVertexLayout(GR_PARAM_A, 28, GR_PARAM_ENABLE); 
  /* The documentation would have you believe that this should be 
   * GR_PARAM_W but it shouldn't */
  grVertexLayout(GR_PARAM_Q, 32, GR_PARAM_ENABLE); 
  grVertexLayout(GR_PARAM_ST0, 36, GR_PARAM_ENABLE);

  /* Get the farthest W value */
  GR_WDEPTHVALUE_FARTHEST = 0xffff;
#endif


  /* Setup rasterization state */
  grClipWindow (0, 0, GLIDE_WIDTH, GLIDE_HEIGHT);
  

  grTexFilterMode (GR_TMU0, 
		   GR_TEXTUREFILTER_BILINEAR, 
		   GR_TEXTUREFILTER_BILINEAR);
  /* Decal Texturing */
  grTexCombine (GR_TMU0, 
		GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
		GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
		FXFALSE, FXFALSE);

  grColorCombine (GR_COMBINE_FUNCTION_SCALE_OTHER, GR_COMBINE_FACTOR_ONE,
		  GR_COMBINE_LOCAL_NONE, GR_COMBINE_OTHER_TEXTURE,
		  FXFALSE);

  /* Enable w buffering */
  grDepthBufferMode (GR_DEPTHBUFFER_WBUFFER);
  grDepthMask (FXTRUE);
}

void
shutdown_glide ()
{
#ifndef USE_GLIDE3
  grSstWinClose ();
#endif
  grGlideShutdown ();  
}

void
load_a_texture (Demo_State *state, char *name, GrTexInfo *grInfo, FxU32 *addr)
{
  Gu3dfInfo info;
  FxU32 size;

  gu3dfGetInfo (name, &info);
  info.data = malloc (info.mem_required);
  gu3dfLoad (name, &info);

  /* Build a GrTexInfo */
#ifndef USE_GLIDE3
  grInfo->smallLod = info.header.small_lod;
  grInfo->largeLod = info.header.large_lod;
  grInfo->aspectRatio = info.header.aspect_ratio;
  grInfo->format = info.header.format;
  grInfo->data = info.data;

  size = grTexCalcMemRequired (info.header.small_lod,
			       info.header.large_lod,
			       info.header.aspect_ratio,
			       info.header.format);
#else
  grInfo->smallLodLog2 = info.header.small_lod;
  grInfo->largeLodLog2 = info.header.large_lod;
  grInfo->aspectRatioLog2 = info.header.aspect_ratio;
  grInfo->format = info.header.format;
  grInfo->data = info.data;

  size = grTexTextureMemRequired (GR_MIPMAPLEVELMASK_BOTH, grInfo);

#endif

  
  if (state->texloc + size < state->texmax)
  {
    grTexDownloadMipMap (GR_TMU0, state->texloc, GR_MIPMAPLEVELMASK_BOTH,
		       grInfo);
    *addr = state->texloc;

    state->texloc += size;
  }
  else
  {
    printf ("failed to download %s: no room size = %ld, texloc = %ld\n", 
	    name, size, state->texloc);
  }

  free (info.data);
  grInfo->data = NULL;
}

void
init_textures (Demo_State *state)
{
  int i;
  char name[100];

  /* Get the texture mem setup */
  state->texmin = grTexMinAddress(GR_TMU0);
  state->texmax = grTexMaxAddress(GR_TMU0);
  state->texloc = state->texmin;

  /* Now load the background textures */
  for (i = 0; i < NTHUMBS; i++)
  {
    sprintf (name, "%s.3df", bg_names[i]);
    load_a_texture (state, name, &(state->background[i]), 
		    &(state->background_tex[i]));
  }

#ifdef DEBUG
  /* Load a set of numbers to palce over windows to make their order
     apparent */
  for (i = 0; i < 10; i++)
  {
    sprintf (name, "data/numbers/%d.3df", i);
    load_a_texture (state, name, &state->numbers[i], 
		    &state->numbers_tex[i]);
  }
#endif
}

/* This is the global data for the wave surface */
GrVertex wave [(WAVE_WIDTH + 1) * (WAVE_HEIGHT + 1)];

/* This function build the geometry for the wave background */
void
init_wave (Demo_State *state)
{
  int i, j;
  FxU32 offset = 0;
  float dx = (DESKTOP_WIDTH)  / (float)(WAVE_WIDTH);
  float dnx = 2 * M_PI / (float)(WAVE_WIDTH);
  float dy = (DESKTOP_HEIGHT) / (float)(WAVE_HEIGHT);
  float dny = 2 * M_PI / (float)(WAVE_HEIGHT);
  float ds = 2 * state->background_texture_half_width / (float)(WAVE_WIDTH);
  float dt = 2 * state->background_texture_half_height / (float)(WAVE_HEIGHT);
  
  float x;
  float nx;
  float y = 0;
  float ny = 0;
  float s;
  float t = 0;

  state->one_over_desktop_w = 1.f / DESKTOP_DEPTH;

  /* Build the geometry */
  for (j = 0; j <= WAVE_HEIGHT; j++)
  {
    x = 0;
    nx = 0;
    s = 0;
    for (i = 0; i <= WAVE_WIDTH; i++)
    {
      wave[offset].x = x;
      wave[offset].y = y;

      /* There is no oow yet.  This is computed per vertex each frame
       * based on the time.
       */

      /*oow = state->one_over_desktop_w;*/
      wave[offset].tmuvtx[0].sow = s * state->one_over_desktop_w;
      wave[offset].tmuvtx[0].tow = t * state->one_over_desktop_w;

      x += dx;      
      nx += dnx;
      s += ds;

      offset++;
    }
    y += dy;
    ny += dny;
    t += dt;
  }
}



