/*
 *  desktop -- The 3dfx Desktop Demo 
 *  COPYRIGHT 3DFX INTERACTIVE, INC. 1999
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "spider.h"

void
put_spider_in_window (Demo_State *state, Window_State *ws)
{
  
  /* Put a spider on the window */
  ws->spider_frame = SPIDER_MIN_FRAME;
  ws->spider_state = SPIDER_WALK;
  
  /* Put the spider in the center of the longest edge.  This makes
   * the radius of its orbit fit in the small direction. */
  /* This is window relative */
  
  if ( (ws->world_lower_right[0] - ws->world_upper_left[0]) >
       (ws->world_upper_left[1] - ws->world_lower_right[1]) )
  {
    /* wide window */
    ws->spider_position[0] = 
      (ws->world_upper_left[1]-ws->world_lower_right[1])/2.0;
    /* Move the spider into the window just a little bit */
    ws->spider_position[1] = -8;
    ws->spider_position[2] = 0;	

    ws->spider_velocity[0] = -0.8;
    ws->spider_velocity[1] = 0;
    ws->spider_velocity[2] = 0;
  }
  else
  {
    /* tall window */
    
    /* Move the spider into the window just a little bit */
    ws->spider_position[0] = 8;	
    ws->spider_position[1] =
      -(ws->world_upper_left[1]-ws->world_lower_right[1])/2.0;
    ws->spider_position[2] = 0;	
    
    ws->spider_velocity[0] = 0;
	ws->spider_velocity[1] = -0.8;
	ws->spider_velocity[2] = 0;
  }  
}

void
update_spider (Demo_State *state, Window_State *ws)
{
  Vector center, delta;
  int i;
  Vector vnorm, z = {0, 0, 1, 0}, neg_v_cross_z;

  /* Note: all of the position, velocity, acceleration
   * calculations are done in X and Y only.  This
   * way the spider never moves into or out of the window. 
   */
  /* For now we orbit the center of the window */

  /* Find the center of the window */
  for (i = 0; i < 2; i++)
  {
    center[i] = (ws->world_upper_left[i] + ws->world_lower_right[i]) / 2.f;

    /* Build the vector toward the attractor */
    delta[i] = center[i] - (ws->spider_position[i] + ws->world_upper_left[i]);  
  }

  /* Set z = 0 for the normalize */
  delta[2] = 0;
  /* Get a unit vector */
  Normalize (delta);

  /* To keep speeds reasonable the attractors have no distance based 
   * falloff */
  /* Set magnitude based on attractor intensity */
  delta[0] *= ws->spider_center_attraction;
  delta[1] *= ws->spider_center_attraction;

  /* FIXME: clamp the force to reasonable max and min */

  /* delta is now the force so update the velocity */
  for (i = 0; i < 2; i++)
  {
    ws->spider_velocity[i] += delta[i] * state->dt;
    ws->spider_position[i] += ws->spider_velocity[i] * state->dt;
  }

  /* I'm tired and haven't come up with a really good way to
   * make the spiders move.  I have however seen the current model
   * position a spider at some illegal vertex position that hung
   * the machine.  If the spiders position is ever outside of its
   * window I am going to warp it back to its initial state.  This
   * won't look nice and it won't force the entire spider to be
   * inside the window but it won't hang the machine either.
   */
  if (ws->spider_position[0] < 0)
  {
#ifdef DEBUG
    printf ("too far left\n");
#endif
    put_spider_in_window (state, ws);
  }
  else if (ws->spider_position[1] > 0)
  {
#ifdef DEBUG
    printf ("too far up\n");
#endif
    put_spider_in_window (state, ws);
  }
  else if (ws->spider_position[0] > ws->world_lower_right[0] - 
	   ws->world_upper_left[0])
  {
#ifdef DEBUG
    printf ("too far right\n");
#endif
    put_spider_in_window (state, ws);
  }
  else if (ws->spider_position[1] > ws->world_upper_left[1] - 
	   ws->world_lower_right[1])
  {
#ifdef DEBUG
    printf ("too far down\n");
#endif
    put_spider_in_window (state, ws);
  }


  /* Now build the alignment matrix */
  /* In standard position the spider faces down the -y axis, the z axis
   * comes out from him and the x axis is perpendicular to both of those.
   * Note all vectors below should be normalized before use
   * So I need a matrix that takes 
   *   ( 0, -1,  0) --> velocity
   *   ( 0,  0,  1) --> ( 0,  0,  1)
   *   ( 1,  0,  0) --> velocity X z
   * Equivalently I can take 
   *   ( 0,  1,  0) --> -velocity
   *   ( 0,  0,  1) --> ( 0,  0,  1)
   *   ( 1,  0,  0) --> (-velocity_dir) X z   
   */ 

  /* get -v */
  vnorm[0] = -ws->spider_velocity[0];
  vnorm[1] = -ws->spider_velocity[1];
  vnorm[2] = 0;
  /* Normalize */
  Normalize (vnorm);

  /* Cross with z axis */
  CrossProduct (neg_v_cross_z, vnorm, z);
  Normalize (neg_v_cross_z);

  /* Build the matrix */
  ws->spider_align[0][0] = neg_v_cross_z[0];
  ws->spider_align[1][0] = neg_v_cross_z[1];
  ws->spider_align[2][0] = neg_v_cross_z[2];
  ws->spider_align[3][0] = 0;

  ws->spider_align[0][1] = vnorm[0];
  ws->spider_align[1][1] = vnorm[1];
  ws->spider_align[2][1] = vnorm[2];
  ws->spider_align[3][1] = 0;

  ws->spider_align[0][2] = z[0];
  ws->spider_align[1][2] = z[1];
  ws->spider_align[2][2] = z[2];
  ws->spider_align[3][2] = 0;

  ws->spider_align[0][3] = 0;
  ws->spider_align[1][3] = 0;
  ws->spider_align[2][3] = 0;
  ws->spider_align[3][3] = 1;  
}

