/*
Donut Bump Mapping Demo
This demo shows how to use a bump mapping technique using Glide(tm)
Copyright (C) 1999  3Dfx Interactive, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "basics.h"
#include "texcache.h"
#include "scrncptr.h"


static struct _texture
{
	FxU16 width;
	FxU16 height;
	void *data_24bpp;
	void *data_16bpp;
} gScreenTexture = {0, 0, NULL, NULL};
static int gX0, gY0;

FxBool SetScreenCaptureRect(int x, int y, int width, int height)
{
	gX0 = x;
	gY0 = y;

	gScreenTexture.width = width;
	gScreenTexture.height = height;

	if (gScreenTexture.data_24bpp)
	{
		delete [] gScreenTexture.data_24bpp;
	}
	gScreenTexture.data_24bpp = new FxU8[3*gScreenTexture.width*gScreenTexture.height];
	if (!gScreenTexture.data_24bpp)
	{
		return FXFALSE;
	}

	if (gScreenTexture.data_16bpp)
	{
		delete [] gScreenTexture.data_16bpp;
	}
	gScreenTexture.data_16bpp = new FxU8[2*gScreenTexture.width*gScreenTexture.height];
	if (!gScreenTexture.data_16bpp)
	{
		return FXFALSE;
	}

	return FXTRUE;
}

void CleanupScreenCapture()
{
	if (gScreenTexture.data_24bpp)
	{
		delete [] gScreenTexture.data_24bpp;
		gScreenTexture.data_24bpp = NULL;
	}
	if (gScreenTexture.data_16bpp)
	{
		delete [] gScreenTexture.data_16bpp;
		gScreenTexture.data_16bpp = NULL;
	}
}

FxBool ScreenCaptureTGA(char *filename)
{
	FxU32 x, y, size;
	FxU16 *src_data_16bpp, *dest_data_16bpp, color;
	FxU8 *dest_data_24bpp;
	FILE *stream;
	TGA_Header header;

	if (!gScreenTexture.data_24bpp || !gScreenTexture.data_16bpp)
	{
		return FXFALSE;
	}

	if (!(stream = fopen(filename, "wb")))
	{
		return FXFALSE;
	}

	// read in the screen pixels
	if (!grLfbReadRegion(GR_BUFFER_BACKBUFFER,
											 gX0, gY0, gScreenTexture.width, gScreenTexture.height,
											 2*gScreenTexture.width, // stride in bytes (of dest data)
											 gScreenTexture.data_16bpp))
	{
		return FXFALSE;
	}

	// fill in the tga header
	header.id_length = 0;
	header.b_color_mapped = 0;
	header.image_type = TGA_TRUE;
	header.color_map_start_lo = 0;
	header.color_map_start_hi = 0;
	header.color_map_entries_lo = 0;
	header.color_map_entries_hi = 0;
	header.color_map_entry_size = 0;
	header.x_origin_lo = 0;
	header.x_origin_hi = 0;
	header.y_origin_lo = 0;
	header.y_origin_hi = 0;
	header.width_lo = (FxU8)(gScreenTexture.width & 0xff);
	header.width_hi = (FxU8)(gScreenTexture.width>>8);
	header.height_lo = (FxU8)(gScreenTexture.height & 0xff);
	header.height_hi = (FxU8)(gScreenTexture.height>>8);
	header.bpp = 24;
	header.reserved = 0;

	if (fwrite(&header, sizeof(TGA_Header), 1, stream) != 1)
	{
		fclose(stream);
		return FXFALSE;
	}

	// flip texture upside down, since the screen is upside down
	dest_data_16bpp = &((FxU16 *)gScreenTexture.data_16bpp)[0];
	src_data_16bpp = &((FxU16 *)gScreenTexture.data_16bpp)[(gScreenTexture.height-1)*gScreenTexture.width];
	for (y=gScreenTexture.height>>1; y>0; y--)
	{
		for (x=gScreenTexture.width; x>0; x--)
		{
			// swap
			color = *dest_data_16bpp;
			*dest_data_16bpp = *src_data_16bpp;
			*src_data_16bpp = color;

			dest_data_16bpp++;
			src_data_16bpp++;
		}
		src_data_16bpp -= gScreenTexture.width + gScreenTexture.width;
	}

	// convert the 565 format to 888 format
	dest_data_24bpp = &((FxU8 *)gScreenTexture.data_24bpp)[0];
	src_data_16bpp = &((FxU16 *)gScreenTexture.data_16bpp)[0];
	for (x=gScreenTexture.width*gScreenTexture.height; x>0; x--)
	{
		// convert 565 to 888
		*dest_data_24bpp = (*src_data_16bpp & 0x001f)<<3;
		dest_data_24bpp++;
		*dest_data_24bpp = (*src_data_16bpp & 0x07e0)>>3;
		dest_data_24bpp++;
		*dest_data_24bpp = (*src_data_16bpp & 0xf800)>>8;
		dest_data_24bpp++;

		src_data_16bpp++;
	}

	size = gScreenTexture.width*gScreenTexture.height;
	if (fwrite(gScreenTexture.data_24bpp, 3, size, stream) != size)
	{
		fclose(stream);
		return FXFALSE;
	}

	fclose(stream);

	return FXTRUE;
}
