/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

#include "config.h"
#include <stdio.h>

#include <gtk/gtk.h>

static void
compare_pixbufs (GdkPixbuf *pixbuf, GdkPixbuf *compare, const gchar *file_type)
{
        if ((gdk_pixbuf_get_width (pixbuf) !=
             gdk_pixbuf_get_width (compare)) ||
            (gdk_pixbuf_get_height (pixbuf) !=
             gdk_pixbuf_get_height (compare)) ||
            (gdk_pixbuf_get_n_channels (pixbuf) !=
             gdk_pixbuf_get_n_channels (compare)) ||
            (gdk_pixbuf_get_has_alpha (pixbuf) !=
             gdk_pixbuf_get_has_alpha (compare)) ||
            (gdk_pixbuf_get_bits_per_sample (pixbuf) !=
             gdk_pixbuf_get_bits_per_sample (compare))) {
                fprintf (stderr,
                         "saved %s file differs from copy in memory\n",
                         file_type);
        } else {
                guchar *orig_pixels;
                guchar *compare_pixels;
                gint    orig_rowstride;
                gint    compare_rowstride;
                gint    width;
                gint    height;
                gint    bytes_per_pixel;
                gint    x, y;
                guchar *p1, *p2;
                gint    count = 0;

                orig_pixels = gdk_pixbuf_get_pixels (pixbuf);
                compare_pixels = gdk_pixbuf_get_pixels (compare);

                orig_rowstride = gdk_pixbuf_get_rowstride (pixbuf);
                compare_rowstride = gdk_pixbuf_get_rowstride (compare);

                width = gdk_pixbuf_get_width (pixbuf);
                height = gdk_pixbuf_get_height (pixbuf);

                /*  well...  */
                bytes_per_pixel = gdk_pixbuf_get_n_channels (pixbuf);

                p1 = orig_pixels;
                p2 = compare_pixels;

                for (y = 0; y < height; y++) {
                        for (x = 0; x < width * bytes_per_pixel; x++)
                                count += (*p1++ != *p2++);

                        orig_pixels += orig_rowstride;
                        compare_pixels += compare_rowstride;

                        p1 = orig_pixels;
                        p2 = compare_pixels;
                }

                if (count > 0) {
                        fprintf (stderr,
                                 "saved %s file differs from copy in memory\n",
                                 file_type);
                }
        }
}

static void
keypress_check (GtkWidget *widget, GdkEventKey *evt, gpointer data)
{
        GdkPixbuf *pixbuf;
        GtkDrawingArea *da = (GtkDrawingArea*)data;
        GError *err = NULL;
        
        pixbuf = (GdkPixbuf *) g_object_get_data (G_OBJECT (da), "pixbuf");

        if (evt->keyval == 'q')
                gtk_main_quit ();
        if (evt->keyval == 's') {
                if (pixbuf == NULL) {
                        fprintf (stderr, "PIXBUF NULL\n");
                        return;
                }	

                if (!gdk_pixbuf_save (pixbuf, "foo.jpg", "jpeg",
                                      &err,
                                      "quality", "100",
                                      NULL)) {
                        fprintf (stderr, "%s", err->message);
                        g_error_free (err);
                } else {
                        GdkPixbuf *compare;

                        compare = gdk_pixbuf_new_from_file ("foo.jpg", &err);

                        if (!compare) {
                                fprintf (stderr, "%s", err->message);
                                g_error_free (err);
                        } else {
                                compare_pixbufs (pixbuf, compare, "jpeg");
                                g_object_unref (compare);
                        }
                                        
                }
        } else if (evt->keyval == 'p') {
                if (pixbuf == NULL) {
                        fprintf (stderr, "PIXBUF NULL\n");
                        return;
                }

                if (!gdk_pixbuf_save (pixbuf, "foo.png", "png", 
                                      &err,
                                      "tEXt::Software", "testpixbuf-save",
                                      NULL)) {
                        fprintf (stderr, "%s", err->message);
                        g_error_free (err);
                } else {
                        GdkPixbuf *compare;

                        compare = gdk_pixbuf_new_from_file ("foo.png", &err);

                        if (!compare) {
                                fprintf (stderr, "%s", err->message);
                                g_error_free (err);
                        } else {
                                compare_pixbufs (pixbuf, compare, "png");
                                g_object_unref (compare);
                        }
                                        
                }
        } else if (evt->keyval == 'a') {
                if (pixbuf == NULL) {
                        fprintf (stderr, "PIXBUF NULL\n");
                        return;
                } else {
                        GdkPixbuf *alpha_buf;

                        alpha_buf = gdk_pixbuf_add_alpha (pixbuf,
                                                          FALSE, 0, 0, 0);

                        g_object_set_data_full (G_OBJECT (da),
                                                "pixbuf", alpha_buf,
                                                (GDestroyNotify) g_object_unref);
                }
        }
}


static int
close_app (GtkWidget *widget, gpointer data)
{
        gtk_main_quit ();
        return TRUE;
}

static int
expose_cb (GtkWidget *drawing_area, GdkEventExpose *evt, gpointer data)
{
        GdkPixbuf *pixbuf;
         
        pixbuf = (GdkPixbuf *) g_object_get_data (G_OBJECT (drawing_area),
						  "pixbuf");
        if (gdk_pixbuf_get_has_alpha (pixbuf)) {
                gdk_draw_rgb_32_image (drawing_area->window,
                                       drawing_area->style->black_gc,
                                       evt->area.x, evt->area.y,
                                       evt->area.width,
                                       evt->area.height,
                                       GDK_RGB_DITHER_MAX,
                                       gdk_pixbuf_get_pixels (pixbuf) +
                                       (evt->area.y * gdk_pixbuf_get_rowstride (pixbuf)) +
                                       (evt->area.x * gdk_pixbuf_get_n_channels (pixbuf)),
                                       gdk_pixbuf_get_rowstride (pixbuf));
        } else {
                gdk_draw_rgb_image (drawing_area->window, 
                                    drawing_area->style->black_gc, 
                                    evt->area.x, evt->area.y,
                                    evt->area.width,
                                    evt->area.height,  
                                    GDK_RGB_DITHER_NORMAL,
                                    gdk_pixbuf_get_pixels (pixbuf) +
                                    (evt->area.y * gdk_pixbuf_get_rowstride (pixbuf)) +
                                    (evt->area.x * gdk_pixbuf_get_n_channels (pixbuf)),
                                    gdk_pixbuf_get_rowstride (pixbuf));
        }
        return FALSE;
}

static int
configure_cb (GtkWidget *drawing_area, GdkEventConfigure *evt, gpointer data)
{
        GdkPixbuf *pixbuf;
                           
        pixbuf = (GdkPixbuf *) g_object_get_data (G_OBJECT (drawing_area),   
						  "pixbuf");
    
        g_print ("X:%d Y:%d\n", evt->width, evt->height);
        if (evt->width != gdk_pixbuf_get_width (pixbuf) || evt->height != gdk_pixbuf_get_height (pixbuf)) {
                GdkWindow *root;
                GdkPixbuf *new_pixbuf;

                root = gdk_get_default_root_window ();
                new_pixbuf = gdk_pixbuf_get_from_drawable (NULL, root, NULL,
                                                           0, 0, 0, 0, evt->width, evt->height);
                g_object_set_data_full (G_OBJECT (drawing_area), "pixbuf", new_pixbuf,
                                        (GDestroyNotify) g_object_unref);
        }

        return FALSE;
}

int
main (int argc, char **argv)
{   
        GdkWindow     *root;
        GtkWidget     *window;
        GtkWidget     *vbox;
        GtkWidget     *drawing_area;
        GdkPixbuf     *pixbuf;    
   
        gtk_init (&argc, &argv);   

        gtk_widget_set_default_colormap (gdk_rgb_get_colormap ());

        root = gdk_get_default_root_window ();
        pixbuf = gdk_pixbuf_get_from_drawable (NULL, root, NULL,
                                               0, 0, 0, 0, 150, 160);
   
        window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
        g_signal_connect (window, "delete_event",
			  G_CALLBACK (close_app), NULL);
        g_signal_connect (window, "destroy",   
			  G_CALLBACK (close_app), NULL);
   
        vbox = gtk_vbox_new (FALSE, 0);
        gtk_container_add (GTK_CONTAINER (window), vbox);  
   
        drawing_area = gtk_drawing_area_new ();
        gtk_widget_set_size_request (GTK_WIDGET (drawing_area),
                                     gdk_pixbuf_get_width (pixbuf),
                                     gdk_pixbuf_get_height (pixbuf));
        g_signal_connect (drawing_area, "expose_event",
			  G_CALLBACK (expose_cb), NULL);

        g_signal_connect (drawing_area, "configure_event",
			  G_CALLBACK (configure_cb), NULL);
        g_signal_connect (window, "key_press_event", 
			  G_CALLBACK (keypress_check), drawing_area);    
        g_object_set_data_full (G_OBJECT (drawing_area), "pixbuf", pixbuf,
                                (GDestroyNotify) g_object_unref);
        gtk_box_pack_start (GTK_BOX (vbox), drawing_area, TRUE, TRUE, 0);
   
        gtk_widget_show_all (window);
        gtk_main ();
        return 0;
}
